const Contest = require("../../Model/Admin/Writer_Contest_Managment_Model");
const Series = require("../../Model/Series_Model");
const User = require("../../Model/User_Model");

// Create new contest
const createContest = async (req, res) => {
  try {
    const contest = new Contest({
      ...req.body,
      rules: req.body.rules.map((rule) => ({
        heading: rule.heading,
        descriptions: Array.isArray(rule.descriptions)
          ? rule.descriptions
          : [rule.descriptions],
      })),
    });
    await contest.save();
    res.status(201).json({
      success: true,
      contest,
      message: "Contest created successfully",
    });
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};

// Get all contests for admin
const getAllContests = async (req, res) => {
  try {
    const contests = await Contest.find().populate(
      "participants",
      "username email"
    );
    res.status(200).json({
      success: true,
      contests,
    });
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};

// Update contest status and visibility
const updateContestStatus = async (req, res) => {
  try {
    const { contestId, status, isVisible } = req.body;
    const contest = await Contest.findByIdAndUpdate(
      contestId,
      { status, isVisible },
      { new: true }
    );
    res.status(200).json({
      success: true,
      contest,
      message: `Contest ${status} successfully`,
    });
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};

// Update Contest
const updateContest = async (req, res) => {
  try {
    const updatedContest = await Contest.findByIdAndUpdate(
      req.params.contestId,
      {
        ...req.body,
        rules: req.body.rules.map((rule) => ({
          heading: rule.heading,
          descriptions: Array.isArray(rule.descriptions)
            ? rule.descriptions
            : [rule.descriptions],
        })),
      },
      { new: true }
    );
    res.status(200).json({
      success: true,
      contest: updatedContest,
      message: "Contest updated successfully",
    });
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};

// Delete contest

const deleteContest = async (req, res) => {
  try {
    await Contest.findByIdAndDelete(req.params.contestId);
    res.status(200).json({
      success: true,
      message: "Contest deleted successfully",
    });
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};

// Get active contests for users
const getActiveContests = async (req, res) => {
  try {
    const contests = await Contest.find({
      status: { $regex: new RegExp("active", "i") }, // Case insensitive match
      isVisible: true,
    }).select("-__v");

    console.log("Found active contests:", contests.length);

    res.status(200).json({
      success: true,
      contests,
    });
  } catch (error) {
    console.error("Error in getActiveContests:", error);
    res.status(500).json({ error: error.message });
  }
};

// Join contest
const joinContest = async (req, res) => {
  try {
    const contest = await Contest.findById(req.params.contestId);
    if (!contest) {
      return res.status(404).json({ error: "Contest not found" });
    }

    if (contest.participants.includes(req.userId)) {
      return res.status(400).json({ error: "Already joined this contest" });
    }

    contest.participants.push(req.userId);
    await contest.save();

    res.status(200).json({
      success: true,
      message: "Successfully joined the contest",
    });
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};

const getContestSubmissions = async (req, res) => {
    try {
        const contestId = req.params.contestId;
        
        // First, get the contest to access its details
        const contest = await Contest.findById(contestId);
        if (!contest) {
            return res.status(404).json({ error: "Contest not found" });
        }
        
        // Get all participants' user IDs
        const participantIds = contest.participants || [];
        
        if (participantIds.length === 0) {
            return res.status(200).json({
                success: true,
                contest: {
                    title: contest.title,
                    description: contest.description,
                    startDate: contest.startDate,
                    endDate: contest.endDate,
                    status: contest.status,
                    participants: []
                },
                submissions: []
            });
        }
        
        // Find all series created by participants that have the contest title in their categories
        const submissions = await Series.find({
            author: { $in: participantIds },
            categories: { $regex: new RegExp(contest.title, 'i') } // Case-insensitive match for contest title
        }).populate('author', 'username email profileImage');
        
        // If no submissions found with the exact title, try finding any series by participants
        // during the contest period
        if (submissions.length === 0) {
            const dateRangeSubmissions = await Series.find({
                author: { $in: participantIds },
                createdAt: { 
                    $gte: new Date(contest.startDate), 
                    $lte: new Date(contest.endDate) 
                }
            }).populate('author', 'username email profileImage');
            
            if (dateRangeSubmissions.length > 0) {
                return res.status(200).json({
                    success: true,
                    contest: {
                        title: contest.title,
                        description: contest.description,
                        startDate: contest.startDate,
                        endDate: contest.endDate,
                        status: contest.status,
                        participants: participantIds
                    },
                    submissions: dateRangeSubmissions
                });
            }
            
            // Last resort: just get any series by the participants
            const anyParticipantSubmissions = await Series.find({
                author: { $in: participantIds }
            }).populate('author', 'username email profileImage')
            .sort({ createdAt: -1 }) // Get the most recent ones
            .limit(50); // Limit to a reasonable number
            
            return res.status(200).json({
                success: true,
                contest: {
                    title: contest.title,
                    description: contest.description,
                    startDate: contest.startDate,
                    endDate: contest.endDate,
                    status: contest.status,
                    participants: participantIds
                },
                submissions: anyParticipantSubmissions,
                note: "Showing recent submissions from participants as no contest-specific submissions were found"
            });
        }
        
        res.status(200).json({
            success: true,
            contest: {
                title: contest.title,
                description: contest.description,
                startDate: contest.startDate,
                endDate: contest.endDate,
                status: contest.status,
                participants: participantIds
            },
            submissions
        });
    } catch (error) {
        console.error("Error fetching contest submissions:", error);
        res.status(500).json({ error: error.message });
    }
}


const getPublicActiveContests = async (req, res) => {
  try {
    console.log("Fetching public active contests...");

    // Use case-insensitive regex to match 'active' or 'ACTIVE'
    const contests = await Contest.find({
      status: { $regex: /^active$/i },
      isVisible: true,
    })
      .sort({ createdAt: -1 })
      .limit(1);

    console.log("Found contests:", contests);

    res.status(200).json({
      success: true,
      contests,
    });
  } catch (error) {
    console.error("Error fetching public contests:", error);
    res.status(500).json({ error: error.message });
  }
};

const getJoinedContests = async (req, res) => {
  try {
    // Find contests where the user is a participant
    const joinedContests = await Contest.find({
      participants: req.userId,
    }).select("_id title");

    res.status(200).json({
      success: true,
      joinedContests,
    });
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};
// Add these new functions to the controller

// Add these new functions to the controller

// Get contest leaderboard
const getContestLeaderboard = async (req, res) => {
    try {
        const contestId = req.params.contestId;
        
        // Get the contest
        const contest = await Contest.findById(contestId);
        if (!contest) {
            return res.status(404).json({ error: "Contest not found" });
        }
        
        // Get all participants' user IDs
        const participantIds = contest.participants || [];
        
        if (participantIds.length === 0) {
            return res.status(200).json({
                success: true,
                contest: {
                    title: contest.title,
                    description: contest.description,
                    startDate: contest.startDate,
                    endDate: contest.endDate,
                    status: contest.status
                },
                leaderboard: []
            });
        }
        
        // Find all series by participants that have the contest title in their categories
        const submissions = await Series.find({
            author: { $in: participantIds },
            categories: { $regex: new RegExp(contest.title, 'i') }
        }).populate('author', 'username email profileImage');
        
        // Calculate scores for each author
        const authorScores = {};
        
        for (const submission of submissions) {
            const authorId = submission.author._id.toString();
            
            // Initialize author score if not exists
            if (!authorScores[authorId]) {
                authorScores[authorId] = {
                    author: {
                        _id: submission.author._id,
                        username: submission.author.username,
                        profileImage: submission.author.profileImage
                    },
                    totalLikes: 0,
                    totalViews: 0,
                    totalRating: 0,
                    submissionCount: 0,
                    overallScore: 0
                };
            }
            
            // Calculate metrics
            const likes = submission.likes?.length || 0;
            const views = submission.episodes?.reduce((sum, ep) => sum + (ep.totalReaders || 0), 0) || 0;
            
            // Calculate average rating
            let avgRating = 0;
            let ratingCount = 0;
            
            if (submission.episodes && submission.episodes.length > 0) {
                const validEpisodes = submission.episodes.filter(ep => ep.averageRating > 0);
                if (validEpisodes.length > 0) {
                    avgRating = validEpisodes.reduce((sum, ep) => sum + ep.averageRating, 0) / validEpisodes.length;
                    ratingCount = validEpisodes.length;
                }
            }
            
            // Update author scores
            authorScores[authorId].totalLikes += likes;
            authorScores[authorId].totalViews += views;
            authorScores[authorId].totalRating += avgRating * ratingCount;
            authorScores[authorId].submissionCount += 1;
        }
        
        // Calculate overall scores
        for (const authorId in authorScores) {
            const score = authorScores[authorId];
            
            // Normalize rating
            const normalizedRating = score.submissionCount > 0 ? 
                score.totalRating / score.submissionCount : 0;
            
            // Calculate overall score (weighted)
            score.overallScore = 
                (score.totalLikes * 0.3) + 
                (score.totalViews * 0.4) + 
                (normalizedRating * 0.3);
            
            // Add normalized rating to the object
            score.averageRating = normalizedRating;
        }
        
        // Convert to array and sort by overall score
        const leaderboard = Object.values(authorScores)
            .sort((a, b) => b.overallScore - a.overallScore)
            .map((entry, index) => ({
                ...entry,
                rank: index + 1
            }));
        
        res.status(200).json({
            success: true,
            contest: {
                _id: contest._id,
                title: contest.title,
                description: contest.description,
                startDate: contest.startDate,
                endDate: contest.endDate,
                status: contest.status
            },
            leaderboard
        });
    } catch (error) {
        console.error("Error generating contest leaderboard:", error);
        res.status(500).json({ error: error.message });
    }
};

// Update the setFeaturedLeaderboard function
const setFeaturedLeaderboard = async (req, res) => {
  try {
    const contestId = req.params.contestId;
    const { action } = req.body; // 'feature' or 'unfeature'
    
    if (!contestId) {
      return res.status(400).json({ 
        success: false,
        error: "Contest ID is required" 
      });
    }
    
    console.log(`Setting featured leaderboard for contest ID: ${contestId}, action: ${action}`);
    
    // Find the contest first to check if it exists
    const contest = await Contest.findById(contestId);
    
    if (!contest) {
      console.error(`Contest not found with ID: ${contestId}`);
      return res.status(404).json({ 
        success: false,
        error: "Contest not found" 
      });
    }
    
    // Determine if we should feature or unfeature
    const shouldFeature = action === 'feature' || 
                         (action !== 'unfeature' && !contest.featuredLeaderboard);
    
    if (shouldFeature) {
      // First, unset any currently featured contest
      await Contest.updateMany(
        { featuredLeaderboard: true },
        { $set: { featuredLeaderboard: false } }
      );
      
      // Then set the selected contest as featured
      contest.featuredLeaderboard = true;
      await contest.save();
      
      console.log(`Successfully set contest "${contest.title}" as featured leaderboard`);
      
      return res.status(200).json({
        success: true,
        message: "Contest set as featured leaderboard",
        contest
      });
    } else {
      // Unfeature the contest
      contest.featuredLeaderboard = false;
      await contest.save();
      
      console.log(`Successfully removed contest "${contest.title}" from featured leaderboard`);
      
      return res.status(200).json({
        success: true,
        message: "Contest removed from featured leaderboard",
        contest
      });
    }
  } catch (error) {
    console.error("Error setting featured leaderboard:", error);
    res.status(500).json({ 
      success: false,
      error: error.message || "Failed to set featured leaderboard" 
    });
  }
};


  

// Add this new function to get top series for the featured leaderboard
const getTopSeriesForLeaderboard = async (req, res) => {
    try {
        // Find the contest with featuredLeaderboard = true
        const featuredContest = await Contest.findOne({ featuredLeaderboard: true });
        
        if (!featuredContest) {
            return res.status(200).json({
                success: true,
                featured: false,
                message: "No featured leaderboard found"
            });
        }
        
        console.log(`Found featured contest: ${featuredContest.title} with ${featuredContest.participants?.length || 0} participants`);
        
        // Get the participant IDs
        const participantIds = featuredContest.participants || [];
        
        if (participantIds.length === 0) {
            return res.status(200).json({
                success: true,
                featured: true,
                contest: {
                    _id: featuredContest._id,
                    title: featuredContest.title,
                    status: featuredContest.status
                },
                topSeries: []
            });
        }
        
        // Find all series by participants - more flexible search approach
        // First try with exact contest title in categories
        let submissions = await Series.find({
            author: { $in: participantIds },
            categories: { $regex: new RegExp(featuredContest.title, 'i') }
        }).populate('author', 'username profileImage');
        
        console.log(`Found ${submissions.length} submissions with exact contest title match`);
        
        // If no submissions found, try with a more flexible approach
        if (submissions.length === 0) {
            // Try to find any submissions by these participants
            submissions = await Series.find({
                author: { $in: participantIds },
                createdAt: { 
                    $gte: new Date(featuredContest.startDate), 
                    $lte: new Date(featuredContest.endDate) 
                }
            }).populate('author', 'username profileImage');
            
            console.log(`Found ${submissions.length} submissions within contest date range`);
            
            // If still no submissions, get any recent submissions from participants
            if (submissions.length === 0) {
                submissions = await Series.find({
                    author: { $in: participantIds }
                }).sort({ createdAt: -1 })
                  .limit(50)
                  .populate('author', 'username profileImage');
                
                console.log(`Found ${submissions.length} recent submissions from participants`);
            }
        }
        
        // Calculate scores for each series
        const seriesScores = submissions.map(series => {
            // Calculate metrics
            const likes = series.likes?.length || 0;
            const views = series.episodes?.reduce((sum, ep) => sum + (ep.totalReaders || 0), 0) || 0;
            
            // Calculate average rating
            let avgRating = 0;
            let ratingCount = 0;
            
            if (series.episodes && series.episodes.length > 0) {
                const validEpisodes = series.episodes.filter(ep => ep.averageRating > 0);
                if (validEpisodes.length > 0) {
                    avgRating = validEpisodes.reduce((sum, ep) => sum + ep.averageRating, 0) / validEpisodes.length;
                    ratingCount = validEpisodes.length;
                }
            }
            
            // Calculate overall score (weighted)
            const overallScore = (likes * 0.3) + (views * 0.4) + (avgRating * 30);
            
            return {
                _id: series._id,
                title: series.title,
                coverImage: series.coverImage,
                summary: series.summary,
                author: {
                    _id: series.author._id,
                    username: series.author.username,
                    profileImage: series.author.profileImage
                },
                totalLikes: likes,
                totalViews: views,
                averageRating: avgRating,
                episodeCount: series.episodes?.length || 0,
                overallScore: overallScore
            };
        });
        
        // Sort by overall score and take top 3
        const topSeries = seriesScores
            .sort((a, b) => b.overallScore - a.overallScore)
            .slice(0, 3)
            .map((series, index) => ({
                ...series,
                rank: index + 1
            }));
        
        res.status(200).json({
            success: true,
            featured: true,
            contest: {
                _id: featuredContest._id,
                title: featuredContest.title,
                description: featuredContest.description,
                status: featuredContest.status
            },
            topSeries
        });
    } catch (error) {
        console.error("Error fetching top series for leaderboard:", error);
        res.status(500).json({ error: error.message });
    }
};

// Modify the existing getFeaturedLeaderboard function to include a type parameter
const getFeaturedLeaderboard = async (req, res) => {
    try {
        const leaderboardType = req.query.type || 'writers'; // Default to writers if not specified
        
        // If type is series, use the series leaderboard function
        if (leaderboardType === 'series') {
            return await getTopSeriesForLeaderboard(req, res);
        }
        
        // Otherwise, continue with the original writers leaderboard logic
        // Find the contest with featuredLeaderboard = true
        const featuredContest = await Contest.findOne({ featuredLeaderboard: true });
        
        if (!featuredContest) {
            return res.status(200).json({
                success: true,
                featured: false,
                message: "No featured leaderboard found"
            });
        }
        
        console.log(`Found featured contest: ${featuredContest.title} with ${featuredContest.participants?.length || 0} participants`);
        
        // Get the leaderboard data
        const participantIds = featuredContest.participants || [];
        
        if (participantIds.length === 0) {
            return res.status(200).json({
                success: true,
                featured: true,
                contest: {
                    _id: featuredContest._id,
                    title: featuredContest.title,
                    status: featuredContest.status
                },
                leaderboard: []
            });
        }
        
        // Find all series by participants - more flexible search approach
        // First try with exact contest title in categories
        let submissions = await Series.find({
            author: { $in: participantIds },
            categories: { $regex: new RegExp(featuredContest.title, 'i') }
        }).populate('author', 'username profileImage');
        
        console.log(`Found ${submissions.length} submissions with exact contest title match`);
        
        // If no submissions found, try with a more flexible approach
        if (submissions.length === 0) {
            // Try to find any submissions by these participants
            submissions = await Series.find({
                author: { $in: participantIds },
                createdAt: { 
                    $gte: new Date(featuredContest.startDate), 
                    $lte: new Date(featuredContest.endDate) 
                }
            }).populate('author', 'username profileImage');
            
            console.log(`Found ${submissions.length} submissions within contest date range`);
            
            // If still no submissions, get any recent submissions from participants
            if (submissions.length === 0) {
                submissions = await Series.find({
                    author: { $in: participantIds }
                }).sort({ createdAt: -1 })
                  .limit(50)
                  .populate('author', 'username profileImage');
                
                console.log(`Found ${submissions.length} recent submissions from participants`);
            }
        }
        
        // Calculate scores for each author
        const authorScores = {};
        
        for (const submission of submissions) {
            const authorId = submission.author._id.toString();
            
            // Initialize author score if not exists
            if (!authorScores[authorId]) {
                authorScores[authorId] = {
                    author: {
                        _id: submission.author._id,
                        username: submission.author.username,
                        profileImage: submission.author.profileImage
                    },
                    totalLikes: 0,
                    totalViews: 0,
                    overallScore: 0
                };
            }
            
            // Calculate metrics
            const likes = submission.likes?.length || 0;
            const views = submission.episodes?.reduce((sum, ep) => sum + (ep.totalReaders || 0), 0) || 0;
            
            // Update author scores
            authorScores[authorId].totalLikes += likes;
            authorScores[authorId].totalViews += views;
        }
        
        // Calculate overall scores
        for (const authorId in authorScores) {
            const score = authorScores[authorId];
            
            // Calculate overall score (simplified)
            score.overallScore = (score.totalLikes * 0.4) + (score.totalViews * 0.6);
        }
        
        // Convert to array and sort by overall score
        const leaderboard = Object.values(authorScores)
            .sort((a, b) => b.overallScore - a.overallScore)
            .slice(0, 3) // Only top 3 for home page
            .map((entry, index) => ({
                ...entry,
                rank: index + 1
            }));
        
        console.log(`Generated leaderboard with ${leaderboard.length} entries (limited to top 3)`);
        
        res.status(200).json({
            success: true,
            featured: true,
            contest: {
                _id: featuredContest._id,
                title: featuredContest.title,
                description: featuredContest.description,
                status: featuredContest.status
            },
            leaderboard
        });
    } catch (error) {
        console.error("Error fetching featured leaderboard:", error);
        res.status(500).json({ error: error.message });
    }
};







// Add this new method to check if a user has joined a contest
const checkContestJoinStatus = async (req, res) => {
  try {
    const contestId = req.params.contestId;
    const userId = req.userId; // From auth middleware
    
    if (!contestId) {
      return res.status(400).json({ 
        success: false, 
        error: "Contest ID is required" 
      });
    }
    
    // Find the contest
    const contest = await Contest.findById(contestId);
    
    if (!contest) {
      return res.status(404).json({ 
        success: false, 
        error: "Contest not found" 
      });
    }
    
    // Check if user has joined this contest
    const hasJoined = contest.participants.includes(userId);
    
    return res.status(200).json({
      success: true,
      hasJoined,
      status: contest.status,
      isActive: contest.status === "active"
    });
  } catch (error) {
    console.error("Error checking contest join status:", error);
    return res.status(500).json({ 
      success: false, 
      error: error.message || "Failed to check contest status" 
    });
  }
};




module.exports = {
  createContest,
  getAllContests,
  updateContestStatus,
  updateContest,
  deleteContest,
  getActiveContests,
  joinContest,
  getContestSubmissions,
  getPublicActiveContests,
  getJoinedContests,
  getContestLeaderboard,
  setFeaturedLeaderboard,
  getFeaturedLeaderboard,
  getTopSeriesForLeaderboard,
  checkContestJoinStatus

};
