const Story = require("../Model/Story_Model");
const Series = require("../Model/Series_Model");
const Category = require("../Model/Admin/Category_admin_Model")


const getCategoryContent = async (req, res) => {
  try {
    const { category } = req.params;
    const { page = 1, limit = 12, contentType, language } = req.query; // Remove shuffle parameter
    
    // Create a case-insensitive regex pattern for the category
    const categoryPattern = new RegExp(category, 'i');
    
    // Calculate skip value for pagination
    const skip = (parseInt(page) - 1) * parseInt(limit);
    
    // Prepare queries based on content type filter
    let storiesQuery = Promise.resolve([]);
    let seriesQuery = Promise.resolve([]);
    
    // Base query objects
    const storyBaseQuery = {
      categories: { $regex: categoryPattern },
      status: "published"
    };
    
    const seriesBaseQuery = {
      categories: { $regex: categoryPattern },
      status: { $ne: "draft" }
    };
    
    // Add language filter if provided
    if (language) {
      storyBaseQuery.language = language;
      seriesBaseQuery.language = language;
    }
    
    // Always use random ordering with MongoDB's aggregation framework
    if (!contentType || contentType === 'all' || contentType === 'stories') {
      storiesQuery = Story.aggregate([
        { $match: storyBaseQuery },
        { $sample: { size: parseInt(limit) / 2 } }, // Random sampling
        { $lookup: {
            from: 'users',
            localField: 'author',
            foreignField: '_id',
            as: 'author'
          }
        },
        { $unwind: '$author' },
        { $project: {
            _id: 1,
            title: 1,
            coverImage: 1,
            categories: 1,
            likes: 1,
            reads: 1,
            averageRating: 1,
            language: 1,
            'author._id': 1,
            'author.username': 1,
            'author.profileImage': 1
          }
        }
      ]);
    }
    
    if (!contentType || contentType === 'all' || contentType === 'series') {
      seriesQuery = Series.aggregate([
        { $match: seriesBaseQuery },
        { $sample: { size: parseInt(limit) / 2 } }, // Random sampling
        { $lookup: {
            from: 'users',
            localField: 'author',
            foreignField: '_id',
            as: 'author'
          }
        },
        { $unwind: '$author' },
        { $project: {
            _id: 1,
            title: 1,
            coverImage: 1,
            categories: 1,
            likes: 1,
            episodes: 1,
            language: 1,
            'author._id': 1,
            'author.username': 1,
            'author.profileImage': 1
          }
        }
      ]);
    }
    
    // Use Promise.all to run queries in parallel
    const [stories, series] = await Promise.all([storiesQuery, seriesQuery]);
    
    // Process stories data - add type field and format data
    const processedStories = stories.map(story => ({
      ...story,
      type: 'story',
      likesCount: Array.isArray(story.likes) ? story.likes.length : 0,
      reads: story.reads || 0
    }));
    
    // Process series data - add type field and calculate total reads
    const processedSeries = series.map(series => {
      // Calculate total reads from episodes
      const totalReads = series.episodes?.reduce(
        (sum, episode) => sum + (episode.reads || 0) + (Array.isArray(episode.readers) ? episode.readers.length : 0), 
        0
      ) || 0;
      
      // Calculate average rating if available
      let averageRating = 0;
      let ratingCount = 0;
      
      if (series.episodes && series.episodes.length > 0) {
        series.episodes.forEach(episode => {
          if (episode.averageRating && !isNaN(episode.averageRating)) {
            averageRating += episode.averageRating;
            ratingCount++;
          }
        });
        
        if (ratingCount > 0) {
          averageRating = averageRating / ratingCount;
        }

      }
      
      return {
        ...series,
        type: 'series',
        reads: totalReads,
        likesCount: Array.isArray(series.likes) ? series.likes.length : 0,
        episodeCount: series.episodes?.length || 0,
        averageRating
      };
    });
    
    // Combine both types of content
    let content = [
      ...processedStories,
      ...processedSeries
    ];
    
    // Shuffle the combined array for extra randomness
    content = shuffleArray(content);
    
    // Check if there are more items to load
    const hasMore = content.length >= parseInt(limit);
    
    // Return the combined content
    res.status(200).json({
      success: true,
      content,
      hasMore,
      filteredBy: language ? { language } : null
    });
  } catch (error) {
    console.error("Error in getCategoryContent:", error);
    // Return empty content instead of error to prevent frontend crashes
    res.status(200).json({ 
      success: true, 
      content: [],
      hasMore: false,
      error: error.message
    });
  }
};

// Helper function to shuffle an array (Fisher-Yates algorithm)
function shuffleArray(array) {
  const newArray = [...array];
  for (let i = newArray.length - 1; i > 0; i--) {
    const j = Math.floor(Math.random() * (i + 1));
    [newArray[i], newArray[j]] = [newArray[j], newArray[i]];
  }
  return newArray;
}






const getActiveCategories = async (req, res) => {
  try {
    console.time('fetchCategories');
    
    // Log the query we're about to execute
    console.log("Executing category query with filter: { isActive: true }");
    
    // Use lean() for better performance and select the title field
    const categories = await Category.find({ isActive: true })
      .select('title coverImage displayOrder showOnHome isActive')
      .sort({ displayOrder: 1 })
      .lean();
    
    console.log("Raw categories from database:", categories);
    console.log("Number of categories found:", categories.length);
    
    // Check if categories have title field
    const hasTitleField = categories.every(cat => cat.title);
    console.log("All categories have title field:", hasTitleField);
    
    if (!hasTitleField) {
      // Log categories without title
      const missingTitles = categories.filter(cat => !cat.title);
      console.log("Categories missing title field:", missingTitles);
    }
    
    console.timeEnd('fetchCategories');
    
    res.status(200).json({
      success: true,
      categories,
      message: "Categories fetched successfully"
    });
  } catch (error) {
    console.error("Error fetching categories:", error);
    res.status(500).json({ 
      success: false, 
      error: error.message || "Failed to fetch categories",
      categories: [] 
    });
  }
};







// Modify the getTrendingContestContent function to sort by views/reads
// Full getTrendingContestContent function without auth requirement
const getTrendingContestContent = async (req, res) => {
  const timerLabel = 'getTrendingContestContent';
  console.time(timerLabel);
  
  try {
    // Get the limit from query params or default to 3
    const limit = parseInt(req.query.limit) || 3;
    
    // Define the specific contest category we want to filter by
    const contestCategory = "Word Weavers Contest S-1";
    
    // Create a case-insensitive regex pattern for more flexible matching
    const categoryPattern = new RegExp(contestCategory, 'i');
    
    console.log(`Searching for series with category matching: ${contestCategory}`);
    
    // Find series that have the specific contest category with optimized query
    // Only select fields we need to improve performance
    const contestSeries = await Series.find({
      status: { $ne: "draft" },
      categories: { $regex: categoryPattern }
    })
    .select('title coverImage author categories episodes.totalReaders episodes.readers episodes.likes episodes.reviews episodes.averageRating likes reads')
    .populate('author', 'username profileImage')
    .lean()
    .exec();
    
    console.log(`Found ${contestSeries.length} contest series`);
    
    // If we have no contest entries, return early with empty array
    if (!contestSeries || contestSeries.length === 0) {
      console.timeEnd(timerLabel);
      return res.status(200).json({
        success: true,
        trendingContent: []
      });
    }
    
    // Calculate metrics for each series with focus on reads/views, likes, and ratings
    const seriesWithMetrics = contestSeries.map(series => {
      // Calculate total reads from episodes
// Update the calculation part in getTrendingContestContent
// Inside the seriesWithMetrics.map function:

// Calculate total reads from episodes
const totalEpisodeReads = series.episodes?.reduce(
  (sum, episode) => {
    // Count both reads and readers array length
    const episodeReadCount = episode?.reads || 0;
    const uniqueReaders = Array.isArray(episode?.readers) ? episode.readers.length : 0;
    const totalEpisodeReads = Math.max(episodeReadCount, uniqueReaders);
    
    // Log episode read counts for debugging
    console.log(`Contest Episode ${episode.title || 'untitled'}: reads=${episodeReadCount}, readers=${uniqueReaders}, total=${totalEpisodeReads}`);
    
    return sum + totalEpisodeReads;
  },
  0
) || 0;

// Use the higher value between calculated reads and stored reads
const totalReads = Math.max(totalEpisodeReads, series.reads || 0);

      // Calculate total likes (series likes + sum of episode likes)
      const seriesLikes = Array.isArray(series.likes) ? series.likes.length : 0;
      const episodeLikes = series.episodes?.reduce(
        (sum, episode) => sum + (Array.isArray(episode?.likes) ? episode.likes.length : 0),
        0
      ) || 0;
      const totalLikes = seriesLikes + episodeLikes;
      
      // Calculate average rating
      let averageRating = 0;
      let totalRatings = 0;
      
      if (series.episodes && series.episodes.length > 0) {
        let ratingSum = 0;
        let ratingCount = 0;
        
        series.episodes.forEach(episode => {
          if (episode.reviews && Array.isArray(episode.reviews)) {
            // Count total ratings across all episodes
            totalRatings += episode.reviews.length;
            
            // Sum up all ratings
            episode.reviews.forEach(review => {
              if (typeof review.rating === 'number' && !isNaN(review.rating)) {
                ratingSum += review.rating;
                ratingCount++;
              }
            });
          } else if (typeof episode.averageRating === 'number' && !isNaN(episode.averageRating)) {
            // If we have averageRating directly
            ratingSum += episode.averageRating;
            ratingCount++;
            
            // Estimate total ratings if we don't have the actual count
            if (episode.reviews?.length) {
              totalRatings += episode.reviews.length;
            }
          }
        });
        
        if (ratingCount > 0) {
          averageRating = ratingSum / ratingCount;
        }
      }
      
      // Calculate a trending score that combines reads, likes, and ratings
      // This gives a balanced view of popularity
      const trendingScore = (totalReads * 1) + (totalLikes * 2) + (averageRating * 10);
      
      return {
        _id: series._id,
        title: series.title,
        coverImage: series.coverImage,
        author: series.author,
        categories: series.categories,
        totalReads: totalReads,
        reads: totalReads, // For compatibility
        likes: series.likes || [],
        likesCount: totalLikes,
        episodeCount: series.episodes?.length || 0,
        averageRating: averageRating.toFixed(1),
        totalRatings: totalRatings,
        type: 'series',
        isContestEntry: true,
        trendingScore: trendingScore
      };
    });
    
    // Sort by trending score in descending order
    seriesWithMetrics.sort((a, b) => b.trendingScore - a.trendingScore);
    
    // Add rank based on sorted position
    const rankedSeries = seriesWithMetrics.map((series, index) => ({
      ...series,
      rank: index + 1
    }));
    
    // Take top entries
    const topSeries = rankedSeries.slice(0, limit);
    
    // Log what we're returning for debugging
    console.log(`Returning ${topSeries.length} trending series sorted by combined score:`, 
      topSeries.map(s => ({
        title: s.title, 
        reads: s.totalReads, 
        likes: s.likesCount,
        rating: s.averageRating
      }))
    );
    
    console.timeEnd(timerLabel);
    
    // Return with a smaller payload to improve performance
    res.status(200).json({
      success: true,
      trendingContent: topSeries.map(series => ({
        _id: series._id,
        title: series.title,
        coverImage: series.coverImage,
        author: series.author,
        totalReads: series.totalReads,
        likesCount: series.likesCount,
        episodeCount: series.episodeCount,
        averageRating: series.averageRating,
        rank: series.rank,
        type: 'series',
        isContestEntry: true
      }))
    });
  } catch (error) {
    console.error("Error in getTrendingContestContent:", error);
    console.timeEnd(timerLabel);
    // Return empty array instead of error
    res.status(200).json({ 
      success: true, 
      trendingContent: [],
      error: "Failed to load trending content"
    });
  }
};


// Optimize getCategoriesForHome function
const getCategoriesForHome = async (req, res) => {
  try {
    console.time('fetchHomeCategories');
    
    // Check if Redis client is available
    let cachedCategories = null;
    if (typeof redisClient !== 'undefined' && redisClient) {
      const cacheKey = 'home_categories';
      cachedCategories = await redisClient.get(cacheKey);
      
      if (cachedCategories) {
        console.log('Serving categories from cache');
        console.timeEnd('fetchHomeCategories');
        return res.status(200).json({
          success: true,
          categories: JSON.parse(cachedCategories),
          message: "Home categories fetched from cache",
        });
      }
    }
    
    // Optimize query with projection and sort by displayOrder
    const categories = await Category.find({
      isActive: true,
      showOnHome: true
    })
    .select('title coverImage displayOrder') // Include displayOrder in selection
    .sort({ displayOrder: 1 }) // Sort by displayOrder in ascending order
    .lean();
    
    // Cache the result if Redis is available
    if (typeof redisClient !== 'undefined' && redisClient) {
      await redisClient.set('home_categories', JSON.stringify(categories), 'EX', 300); // 5 minutes
    }
    
    console.timeEnd('fetchHomeCategories');
    
    res.status(200).json({
      success: true,
      categories,
      message: "Home categories fetched successfully",
    });
  } catch (error) {
    console.error("Error in getCategoriesForHome:", error);
    res.status(200).json({ 
      success: true, 
      categories: [],
      error: "Failed to load categories"
    });
  }
};


// Get all categories
const getAllCategories = async (req, res) => {
  try {
    const categories = await Category.find().sort({ name: 1 }).lean();
    
    res.status(200).json({
      success: true,
      categories,
      message: "Categories fetched successfully"
    });
  } catch (error) {
    console.error("Error fetching categories:", error);
    res.status(200).json({ 
      success: false, 
      categories: [],
      message: "Failed to fetch categories"
    });
  }
};
// Make sure the getContentByCategory function is optimized
const getContentByCategory = async (req, res) => {
  try {
    const { category, limit = 6, language, page = 1 } = req.query;
    
    // If category is missing, return empty content
    if (!category) {
      return res.status(200).json({ 
        success: true, 
        content: [] 
      });
    }

    const skip = (parseInt(page) - 1) * parseInt(limit);
    
    // Create base queries
    const baseQuery = { 
      categories: category,
      status: "published"
    };
    
    const seriesQuery = { 
      categories: category,
      status: { $ne: "draft" }
    };
    
    // Add language filter if provided
    if (language) {
      baseQuery.language = language;
      seriesQuery.language = language;
    }
    
    // Use Promise.all to run queries in parallel with lean() for better performance
    const [stories, series] = await Promise.all([
      // Get stories in this category
      Story.find(baseQuery)
        .select('title coverImage summary author categories likes reads createdAt')
        .populate('author', 'username profileImage')
        .sort({ createdAt: -1 })
        .skip(skip)
        .limit(Math.ceil(parseInt(limit) / 2)) // Half the limit for stories
        .lean(),
        
      // Get series in this category
      Series.find(seriesQuery)
        .select('title coverImage summary author categories episodes likes reads createdAt')
        .populate('author', 'username profileImage')
        .sort({ createdAt: -1 })
        .skip(skip)
        .limit(Math.ceil(parseInt(limit) / 2)) // Half the limit for series
        .lean()
    ]);
    
    // Process stories data
    const processedStories = stories.map(story => ({
      ...story,
      likesCount: Array.isArray(story.likes) ? story.likes.length : 0,
      type: "story"
    }));
    
    // Process series data
    const processedSeries = series.map(series => {
      const episodeCount = series.episodes?.length || 0;
      const totalReads = series.episodes?.reduce(
        (sum, episode) => sum + (episode.reads || 0), 0
      ) || 0;
      
      return {
        ...series,
        totalReads,
        episodeCount,
        likesCount: Array.isArray(series.likes) ? series.likes.length : 0,
        type: "series"
      };
    });
    
    // Combine content
    const combinedContent = [...processedStories, ...processedSeries];
    
    res.status(200).json({
      success: true,
      content: combinedContent
    });
  } catch (error) {
    console.error("Error fetching content by category:", error);
    res.status(200).json({ 
      success: true, 
      content: []
    });
  }
};




// Update the getActiveCategories function to ensure we return title field




// Create a new category
const createCategory = async (req, res) => {
  try {
    const { name, description } = req.body;
    
    // Check if category already exists
    const existingCategory = await Category.findOne({ name });
    if (existingCategory) {
      return res.status(400).json({ 
        success: false, 
        message: "Category already exists" 
      });
    }
    
    const newCategory = new Category({
      name,
      description
    });
    
    await newCategory.save();
    
    res.status(201).json({
      success: true,
      category: newCategory,
      message: "Category created successfully"
    });
  } catch (error) {
    console.error("Error creating category:", error);
    res.status(500).json({ 
      success: false, 
      message: "Failed to create category",
      error: error.message
    });
  }
};


  
  
  
  
  
  
  module.exports = {
    getCategoryContent,
    getActiveCategories,
    getTrendingContestContent,
    getAllCategories,
    getContentByCategory,
    createCategory,
        getCategoriesForHome
  };