const Series = require("../Model/Series_Model");
const User = require("../Model/User_Model"); // Add this import at the top
const UserController = require("../Controller/User_Controller")
const multer = require("multer");
const path = require("path");
const {
  notifyFollowers,
  createNotification,
} = require("../utils/notifications");
const {updateReaderLevel} = require("../utils/readerLevelUtils")
// Configure multer for image upload
const storage = multer.diskStorage({
  destination: "./public/uploads/covers",
  filename: function (req, file, cb) {
    cb(null, "cover-" + Date.now() + path.extname(file.originalname));
  },
});

const upload = multer({
  storage: storage,
  limits: { fileSize: 5000000 }, // 5MB limit
  fileFilter: function (req, file, cb) {
    checkFileType(file, cb);
  },
}).single("coverImage");

// Check file type
function checkFileType(file, cb) {
  const filetypes = /jpeg|jpg|png|gif/;
  const extname = filetypes.test(path.extname(file.originalname).toLowerCase());
  const mimetype = filetypes.test(file.mimetype);

  if (mimetype && extname) {
    return cb(null, true);
  } else {
    cb("Error: Images Only!");
  }
}

// Base Series Functions
const createSeries = async (req, res) => {
  try {
    const {
      title,
      coverImage,
      summary,
      storyType,
      categories,
      language,
      status,
    } = req.body;

    const series = new Series({
      title,
      coverImage,
      summary,
      author: req.userId,
      storyType,
      categories,
      language: language || "EN",
      status: "ongoing",
    });

    const savedSeries = await series.save();

    // Notify followers using the notifyFollowers utility function
    await notifyFollowers(req.userId, savedSeries._id, "Series", title);

    res.status(201).json({
      success: true,
      series: savedSeries,
      message: "Series Created Successfully",
    });
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};

const getUserSeries = async (req, res) => {
  try {
    const series = await Series.find({ author: req.userId })
      .populate("author", "username")
      .populate("readHistory.user", "username")
      .populate("likes", "username") // Add this line to populate likes
      .sort({ createdAt: -1 });

    // Ensure likes is always an array in the response
    const seriesWithLikes = series.map((s) => ({
      ...s.toObject(),
      likes: s.likes || [],
    }));

    res.status(200).json({
      success: true,
      series: seriesWithLikes,
      message: "Series fetched successfully",
    });
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};

const saveDraft = async (req, res) => {
  try {
    const {
      title,
      content,
      coverImage,
      summary,
      storyType,
      categories,
      language,
    } = req.body;

    const draft = new Series({
      title: title || "Untitled Draft",
      coverImage,
      summary,
      author: req.userId,
      storyType: storyType || "Fiction",
      categories: Array.isArray(categories) ? categories : [],
      status: "draft",
      episodes: [
        {
          title: title || "Untitled Chapter",
          content: content || "",
          chapterNumber: 1,
        },
      ],
      language: language || "EN", // Add language here
    });

    const savedDraft = await draft.save();
    res.status(201).json({
      success: true,
      draft: savedDraft,
      message: "Draft saved successfully",
    });
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};

const getDrafts = async (req, res) => {
  try {
    const drafts = await Series.find({
      author: req.userId,
      status: "draft",
    }).sort({ createdAt: -1 });
    res.status(200).json({
      success: true,
      drafts,
      message: "Drafts fetched successfully",
    });
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};

const deleteSeries = async (req, res) => {
  try {
    const series = await Series.findById(req.params.id);
    const user = await User.findById(req.userId);

    // Calculate total words in all episodes
    const totalWords = series.episodes.reduce((sum, episode) => {
      return (
        sum +
        episode.content.split(/\s+/).filter((word) => word.length > 0).length
      );
    }, 0);

    const totalEarnings = totalWords * 0.1;
    const dateKey = series.createdAt.toISOString().split("T")[0];

    // Update daily stats
    await User.findByIdAndUpdate(req.userId, {
      $inc: {
        "writingStats.totalWordCount": -totalWords,
        "writingStats.totalEarnings": -totalEarnings,
        [`writingStats.dailyWordCount.${dateKey}.count`]: -totalWords,
        [`writingStats.dailyWordCount.${dateKey}.earnings`]: -totalEarnings,
      },
    });

    await Promise.all([
      Series.findByIdAndDelete(req.params.id),
      User.updateMany(
        { "notifications.contentId": series._id },
        { $pull: { notifications: { contentId: series._id } } }
      ),
    ]);

    res.status(200).json({
      success: true,
      message: "Series and related data deleted successfully",
    });
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};

const deleteSeriesDraft = async (req, res) => {
  try {
    const draft = await Series.findOneAndDelete({
      _id: req.params.id,
      author: req.userId,
      status: "draft",
    });
    if (!draft) return res.status(404).json({ error: "Draft not found" });
    res.status(200).json({
      success: true,
      message: "Series Draft Deleted Successfully",
    });
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};

const getSeriesById = async (req, res) => {
  try {
    const series = await Series.findById(req.params.id)
      .populate({
        path: "author",
        select: "username profileImage followers following",
        populate: [
          {
            path: "followers",
            select: "_id username",
          },
          {
            path: "following",
            select: "_id username",
          },
        ],
      })
      .populate("reviews.user", "username")
      .populate("reviews.replies.user", "username")
      .populate("likes", "username");

    if (!series) return res.status(404).json({ error: "Series not found" });

    // Calculate stats
    const authorStats = {
      followersCount: series.author.followers.length,
      followingCount: series.author.following.length,
      isFollowing: series.author.followers.some(
        (follower) => follower._id.toString() === req.userId
      ),
    };

    res.status(200).json({
      success: true,
      series,
      authorStats,
      message: "Series Fetched Successfully",
    });
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};

// Episode Functions
const addEpisode = async (req, res) => {
  try {
    const { title, content, chapterNumber } = req.body;
    const user = await User.findById(req.userId);

    const series = await Series.findOne({
      _id: req.params.id,
      author: req.userId,
    });

    if (!series) {
      return res.status(404).json({ error: "Series not found" });
    }

    series.episodes.push({ title, content, chapterNumber });
    await series.save();

    // Update word count stats
    if (series.status !== "draft") {
      await user.updateWordCount(series._id, "episode", content);
    }

    res.status(201).json({
      success: true,
      series,
      message: "Episode Added Successfully",
    });
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};

const getEpisodeById = async (req, res) => {
  try {
    const series = await Series.findById(req.params.id);
    if (!series) return res.status(404).json({ error: "Series not found" });

    const episode = series.episodes.id(req.params.episodeId);
    if (!episode) return res.status(404).json({ error: "Episode not found" });

    const episodeResponse = {
      ...episode.toObject(),
      totalReaders: episode.readers?.length || 0,
      readerCount: episode.readers?.length || 0,
    };

    res.status(200).json({
      success: true,
      episode: episodeResponse,
      series,
    });
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};

const updateEpisode = async (req, res) => {
  try {
    const series = await Series.findOne({
      _id: req.params.id,
      author: req.userId,
    });
    if (!series) return res.status(404).json({ error: "Series not found" });

    const episode = series.episodes.id(req.params.episodeId);
    if (!episode) return res.status(404).json({ error: "Episode not found" });

    // Calculate word count difference
    const oldWordCount = episode.content
      .split(/\s+/)
      .filter((word) => word.length > 0).length;
    const newWordCount = req.body.content
      .split(/\s+/)
      .filter((word) => word.length > 0).length;
    const wordCountDiff = newWordCount - oldWordCount;

    episode.title = req.body.title;
    episode.content = req.body.content;
    episode.updatedAt = Date.now();

    await series.save();

    // Update user's writing stats with the difference
    if (wordCountDiff !== 0) {
      const today = new Date().toISOString().split("T")[0];
      const month = new Date().toISOString().slice(0, 7);

      await User.findByIdAndUpdate(req.userId, {
        $inc: {
          "writingStats.totalWordCount": wordCountDiff,
          "writingStats.earnings": wordCountDiff * 0.1,
          [`writingStats.dailyWordCount.${today}`]: wordCountDiff,
          [`writingStats.monthlyWordCount.${month}`]: wordCountDiff,
        },
      });
    }

    res.status(200).json({
      success: true,
      episode,
      message: "Episode Updated Successfully",
    });
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};

const deleteEpisode = async (req, res) => {
  try {
    const series = await Series.findOne({
      _id: req.params.id,
      author: req.userId,
    });

    if (!series) {
      return res.status(404).json({ error: "Series not found" });
    }

    // Find the episode to be deleted
    const episode = series.episodes.id(req.params.episodeId);
    if (!episode) {
      return res.status(404).json({ error: "Episode not found" });
    }

    // Remove the episode from the series
    series.episodes.pull(req.params.episodeId);
    await series.save();

    res.status(200).json({
      success: true,
      message: "Episode deleted successfully",
    });
  } catch (error) {
    console.error("Error deleting episode:", error);
    res.status(500).json({ error: error.message });
  }
};

// Series Interaction Functions
const likeSeries = async (req, res) => {
  try {
    const series = await Series.findById(req.params.id);
    if (!series) return res.status(404).json({ error: "Series not found" });

    const likeIndex = series.likes.indexOf(req.userId);
    if (likeIndex === -1) {
      series.likes.push(req.userId);

      const user = await User.findById(req.userId);
      await createNotification(series.author, {
        type: "like",
        message: `${user.username} liked your series "${series.title}"`,
        fromUser: req.userId,
        contentId: series._id,
        contentModel: "Series",
      });
    } else {
      series.likes.splice(likeIndex, 1);
    }
    await series.save();
    res.json({
      success: true,
      message: "Series Like Updated Successfully",
      likes: series.likes.length,
      isLiked: likeIndex === -1,
    });
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};

const incrementReads = async (req, res) => {
  try {
    const series = await Series.findById(req.params.id);
    if (!series) return res.status(404).json({ error: "Series not found" });

    const HOURS_24 = 24 * 60 * 60 * 1000;
    const now = new Date();
    const userReadHistory = series.readHistory.find(
      (record) => record.user.toString() === req.userId
    );

    if (!userReadHistory) {
      series.reads += 1;
      series.readHistory.push({
        user: req.userId,
        lastRead: now,
      });
    } else {
      const timeSinceLastRead = now - new Date(userReadHistory.lastRead);
      if (timeSinceLastRead >= HOURS_24) {
        series.reads += 1;
        userReadHistory.lastRead = now;
      }
    }

    await series.save();
    res.status(200).json({
      success: true,
      reads: series.reads,
    });
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};


// Episode Interaction Functions
const likeEpisode = async (req, res) => {
  try {
    const series = await Series.findById(req.params.id);
    if (!series) return res.status(404).json({ error: "Series not found" });

    const episode = series.episodes.id(req.params.episodeId);
    if (!episode) return res.status(404).json({ error: "Episode not found" });

    // Initialize likes array if needed
    if (!Array.isArray(episode.likes)) {
      episode.likes = [];
    }

    const likeIndex = episode.likes.indexOf(req.userId);
    if (likeIndex === -1) {
      episode.likes.push(req.userId);
    } else {
      episode.likes.splice(likeIndex, 1);
    }

    await series.save();

    res.json({
      success: true,
      likes: episode.likes,
      isLiked: likeIndex === -1,
      likesCount: episode.likes.length,
    });
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};

const incrementEpisodeReads = async (req, res) => {
  try {
    const series = await Series.findById(req.params.id);
    if (!series) return res.status(404).json({ error: "Series not found" });

    const episode = series.episodes.id(req.params.episodeId);
    if (!episode) return res.status(404).json({ error: "Episode not found" });

    const HOURS_24 = 24 * 60 * 60 * 1000;
    const now = new Date();

    // Initialize readHistory if it doesn't exist
    if (!episode.readHistory) {
      episode.readHistory = [];
    }

    const existingRead = episode.readHistory.find(
      (record) => record.user.toString() === req.userId
    );

    if (!existingRead) {
      // First time reading
      episode.readHistory.push({
        user: req.userId,
        lastRead: now,
      });
      episode.reads = episode.readHistory.length;
    } else {
      const timeSinceLastRead = now - new Date(existingRead.lastRead);
      if (timeSinceLastRead >= HOURS_24) {
        existingRead.lastRead = now;
        existingRead.readCount += 1;
        episode.reads += 1;
      }
    }

    await series.save();

    res.status(200).json({
      success: true,
      reads: episode.reads,
    });
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};


const likeEpisodeReply = async (req, res) => {
  try {
    const series = await Series.findById(req.params.id);
    if (!series) return res.status(404).json({ error: "Series not found" });

    const episode = series.episodes.id(req.params.episodeId);
    if (!episode) return res.status(404).json({ error: "Episode not found" });

    const review = episode.reviews.id(req.params.reviewId);
    if (!review) return res.status(404).json({ error: "Review not found" });

    const reply = review.replies.id(req.params.replyId);
    if (!reply) return res.status(404).json({ error: "Reply not found" });

    const likeIndex = reply.likes.indexOf(req.userId);
    if (likeIndex === -1) {
      reply.likes.push(req.userId);
    } else {
      reply.likes.splice(likeIndex, 1);
    }

    await series.save();

    const populatedSeries = await Series.findById(series._id)
      .populate("episodes.reviews.user", "username")
      .populate("episodes.reviews.replies.user", "username")
      .populate("episodes.reviews.replies.mentionedUser", "username");

    const updatedEpisode = populatedSeries.episodes.id(req.params.episodeId);

    res.status(200).json({
      success: true,
      reviews: updatedEpisode.reviews,
      message: "Reply like updated successfully",
    });
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};

const updateEpisodeReply = async (req, res) => {
  try {
    const { content } = req.body;
    const series = await Series.findById(req.params.id);
    if (!series) return res.status(404).json({ error: "Series not found" });

    const episode = series.episodes.id(req.params.episodeId);
    if (!episode) return res.status(404).json({ error: "Episode not found" });

    const review = episode.reviews.id(req.params.reviewId);
    if (!review) return res.status(404).json({ error: "Review not found" });

    const reply = review.replies.id(req.params.replyId);
    if (!reply) return res.status(404).json({ error: "Reply not found" });

    if (String(reply.user) !== String(req.userId)) {
      return res
        .status(403)
        .json({ error: "Not authorized to update this reply" });
    }

    reply.content = content;
    await series.save();

    const populatedSeries = await Series.findById(series._id)
      .populate("episodes.reviews.user", "username")
      .populate("episodes.reviews.replies.user", "username")
      .populate("episodes.reviews.replies.mentionedUser", "username");

    const updatedEpisode = populatedSeries.episodes.id(req.params.episodeId);

    res.status(200).json({
      success: true,
      reviews: updatedEpisode.reviews,
      message: "Reply updated successfully",
    });
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};

const deleteEpisodeReply = async (req, res) => {
  try {
    const series = await Series.findById(req.params.id);
    if (!series) return res.status(404).json({ error: "Series not found" });

    const episode = series.episodes.id(req.params.episodeId);
    if (!episode) return res.status(404).json({ error: "Episode not found" });

    const review = episode.reviews.id(req.params.reviewId);
    if (!review) return res.status(404).json({ error: "Review not found" });

    const reply = review.replies.id(req.params.replyId);
    if (!reply) return res.status(404).json({ error: "Reply not found" });

    if (String(reply.user) !== String(req.userId)) {
      return res
        .status(403)
        .json({ error: "Not authorized to delete this reply" });
    }

    review.replies.pull(req.params.replyId);
    await series.save();

    const populatedSeries = await Series.findById(series._id)
      .populate("episodes.reviews.user", "username")
      .populate("episodes.reviews.replies.user", "username");

    const updatedEpisode = populatedSeries.episodes.id(req.params.episodeId);

    res.status(200).json({
      success: true,
      reviews: updatedEpisode.reviews,
      message: "Reply deleted successfully",
    });
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};

// Review Functions
// Add this helper function at the top
const calculateEpisodeAverageRating = (reviews) => {
  if (!reviews?.length) return 0;
  const sum = reviews.reduce((acc, review) => acc + review.rating, 0);
  return Number((sum / reviews.length).toFixed(1));
};

// Update the addEpisodeReview function
// const addEpisodeReview = async (req, res) => {
//   try {
//     const { rating, content } = req.body;
//     if (!content || !content.trim()) {
//       return res.status(400).json({ error: "Review content is required" });
//     }

//     const user = await User.findById(req.userId);
//     const series = await Series.findById(req.params.id);
//     if (!series) return res.status(404).json({ error: "Series not found" });

//     const episode = series.episodes.id(req.params.episodeId);
//     if (!episode) return res.status(404).json({ error: "Episode not found" });

//     // Direct comparison of author ID with reviewer ID
//     const isAuthor = series.author.toString() === req.userId;

//     // Only create notification if reviewer is not the author
//     if (!isAuthor) {
//       await createNotification(series.author, {
//         type: "review",
//         message: `${user.username} reviewed episode "${episode.title}" in your series "${series.title}"`,
//         fromUser: req.userId,
//         contentId: series._id,
//         contentModel: "Series",
//         contentAuthorId: series.author
//       });
//     }

//     // Add the review
//     episode.reviews.push({
//       user: req.userId,
//       rating,
//       content: content.trim(),
//       likes: [],
//       replies: [],
//     });

//     await series.save();

//     // Add notification for episode review
//     await User.findByIdAndUpdate(series.author, {
//       $push: {
//         notifications: {
//           type: "review",
//           message: `${user.username} reviewed episode "${episode.title}" in your series "${series.title}"`,
//           fromUser: req.userId,
//           contentId: series._id,
//           contentModel: "Series",
//           isRead: false,
//           createdAt: new Date(),
//         },
//       },
//     });

//     // Return populated review data
//     const populatedSeries = await Series.findById(series._id)
//       .populate("episodes.reviews.user", "username")
//       .populate("episodes.reviews.replies.user", "username");

//     res.status(201).json({
//       success: true,
//       message: "Review added successfully",
//       series: populatedSeries,
//     });
//   } catch (error) {
//     res.status(500).json({ error: error.message });
//   }
// };

const addEpisodeReview = async (req, res) => {
  try {
    const { rating, content } = req.body;
    if (!content || !content.trim()) {
      return res.status(400).json({ error: "Review content is required" });
    }

    const user = await User.findById(req.userId);
    const series = await Series.findById(req.params.id);
    if (!series) return res.status(404).json({ error: "Series not found" });

    const episode = series.episodes.id(req.params.episodeId);
    if (!episode) return res.status(404).json({ error: "Episode not found" });

    // Direct comparison of author ID with reviewer ID
    const isAuthor = series.author.toString() === req.userId;

    // Only create notification if reviewer is not the author
    if (!isAuthor) {
      await createNotification(series.author, {
        type: "review",
        message: `${user.username} reviewed episode "${episode.title}" in your series "${series.title}"`,
        fromUser: req.userId,
        contentId: series._id,
        contentModel: "Series",
        contentAuthorId: series.author,
      });
    }

    episode.reviews.push({
      user: req.userId,
      rating,
      content: content.trim(),
      likes: [],
      replies: [],
    });

    await series.save();

    const populatedSeries = await Series.findById(series._id)
      .populate("episodes.reviews.user", "username")
      .populate("episodes.reviews.replies.user", "username");

    res.status(201).json({
      success: true,
      message: "Review added successfully",
      series: populatedSeries,
    });
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};

const getEpisodeReviews = async (req, res) => {
  try {
    const series = await Series.findById(req.params.id)
      .populate("episodes.reviews.user", "username")
      .populate("episodes.reviews.replies.user", "username");

    if (!series) return res.status(404).json({ error: "Series not found" });

    const episode = series.episodes.id(req.params.episodeId);
    if (!episode) return res.status(404).json({ error: "Episode not found" });

    res.status(200).json({
      success: true,
      reviews: episode.reviews || [],
      message: "Reviews fetched successfully",
    });
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};

const addEpisodeReviewReply = async (req, res) => {
  try {
    const { content } = req.body;
    const series = await Series.findById(req.params.id);
    if (!series) return res.status(404).json({ error: "Series not found" });

    const episode = series.episodes.id(req.params.episodeId);
    if (!episode) return res.status(404).json({ error: "Episode not found" });

    const review = episode.reviews.id(req.params.reviewId);
    if (!review) return res.status(404).json({ error: "Review not found" });

    const replier = await User.findById(req.userId);
    review.replies.push({
      user: req.userId,
      content,
      likes: [],
    });

    await series.save();

    // Add notification with reviewAuthorId
    await createNotification(review.user, {
      type: "reply",
      message: `${replier.username} replied to your review on episode "${episode.title}"`,
      fromUser: req.userId,
      contentId: series._id,
      contentModel: "Series",
      reviewAuthorId: review.user, // Added this line
    });

    res.status(201).json({
      success: true,
      message: "Reply added successfully",
      series,
    });
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};

const likeEpisodeReview = async (req, res) => {
  try {
    const series = await Series.findById(req.params.id);
    if (!series) return res.status(404).json({ error: "Series not found" });

    const episode = series.episodes.id(req.params.episodeId);
    if (!episode) return res.status(404).json({ error: "Episode not found" });

    const review = episode.reviews.id(req.params.reviewId);
    if (!review) return res.status(404).json({ error: "Review not found" });

    const likeIndex = review.likes.indexOf(req.userId);
    if (likeIndex === -1) {
      review.likes.push(req.userId);
    } else {
      review.likes.splice(likeIndex, 1);
    }

    await series.save();

    const populatedSeries = await Series.findById(series._id)
      .populate("episodes.reviews.user", "username")
      .populate("episodes.reviews.replies.user", "username")
      .populate("episodes.reviews.replies.mentionedUser", "username");

    const updatedEpisode = populatedSeries.episodes.id(req.params.episodeId);

    res.status(200).json({
      success: true,
      reviews: updatedEpisode.reviews,
      message: "Review like updated successfully",
    });
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};

const updateEpisodeReview = async (req, res) => {
  try {
    const { content, rating } = req.body;
    const series = await Series.findById(req.params.id);
    if (!series) return res.status(404).json({ error: "Series not found" });

    const episode = series.episodes.id(req.params.episodeId);
    if (!episode) return res.status(404).json({ error: "Episode not found" });

    const review = episode.reviews.id(req.params.reviewId);
    if (!review) return res.status(404).json({ error: "Review not found" });

    if (String(review.user) !== String(req.userId)) {
      return res
        .status(403)
        .json({ error: "Not authorized to update this review" });
    }

    review.content = content;
    if (rating) review.rating = rating;

    await series.save();

    const populatedSeries = await Series.findById(series._id)
      .populate("episodes.reviews.user", "username")
      .populate("episodes.reviews.replies.user", "username");

    const updatedEpisode = populatedSeries.episodes.id(req.params.episodeId);

    res.status(200).json({
      success: true,
      reviews: updatedEpisode.reviews,
      message: "Review updated successfully",
    });
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};

const deleteEpisodeReview = async (req, res) => {
  try {
    const series = await Series.findById(req.params.id);
    if (!series) return res.status(404).json({ error: "Series not found" });

    const episode = series.episodes.id(req.params.episodeId);
    if (!episode) return res.status(404).json({ error: "Episode not found" });

    const review = episode.reviews.id(req.params.reviewId);
    if (!review) return res.status(404).json({ error: "Review not found" });

    if (String(review.user) !== String(req.userId)) {
      return res
        .status(403)
        .json({ error: "Not authorized to delete this review" });
    }

    episode.reviews.pull(req.params.reviewId);

    // Recalculate episode average rating
    if (episode.reviews.length > 0) {
      const sum = episode.reviews.reduce((acc, rev) => acc + rev.rating, 0);
      episode.averageRating = (sum / episode.reviews.length).toFixed(1);
    } else {
      episode.averageRating = 0;
    }

    await series.save();

    // Return updated series data
    const updatedSeries = await Series.findById(req.params.id)
      .populate("episodes.reviews.user", "username")
      .populate("episodes.reviews.replies.user", "username");

    res.status(200).json({
      success: true,
      message: "Review deleted successfully",
      series: updatedSeries,
      episode: updatedSeries.episodes.id(req.params.episodeId),
    });
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};

// Optimize the getAllPublishedSeries function with better timeout handling
const getAllPublishedSeries = async (req, res) => {
  try {
    console.time("fetchAllSeries");
    const { 
      sortBy = "popular", 
      sortOrder = "desc",
      category = null,
      search = "",
      language = null,
      limit = 100 
    } = req.query;

    // Create base query
    const query = { status: "published" };
    
    // Add language filter
    if (language && language !== "ALL") {
      query.language = language;
    }
    
    // Add category filter if provided
    if (category) {
      query.categories = { $in: [category] };
    }

    // Add search functionality
    if (search && search.trim() !== "") {
      const searchRegex = new RegExp(search, "i");
      query.$or = [
        { title: searchRegex },
        { summary: searchRegex }
      ];
    }
    
    // Determine sort options
    let sortOptions = {};
    if (sortBy === "popular") {
      sortOptions = { reads: sortOrder === "desc" ? -1 : 1 };
    } else if (sortBy === "newest") {
      sortOptions = { createdAt: sortOrder === "desc" ? -1 : 1 };
    } else if (sortBy === "rating") {
      sortOptions = { averageRating: sortOrder === "desc" ? -1 : 1 };
    } else {
      sortOptions = { reads: -1 };
    }
    
    // Add secondary sort
    sortOptions.createdAt = sortOrder === "desc" ? -1 : 1;

    // OPTIMIZATION: Use a timeout promise to prevent hanging
    const timeoutPromise = new Promise((_, reject) => 
      setTimeout(() => reject(new Error('MongoDB query timeout')), 5000)
    );
    
    // OPTIMIZATION: Only select necessary fields and use lean() for better performance
    const queryPromise = Series.find(query)
      .select("title coverImage summary author categories likes reads createdAt episodeCount averageRating language")
      .populate("author", "username")
      .sort(sortOptions)
      .limit(parseInt(limit))
      .lean()
      .maxTimeMS(5000); // Add maxTimeMS to MongoDB query
    
    // Race the query against the timeout
    const series = await Promise.race([queryPromise, timeoutPromise])
      .catch(err => {
        console.error("Error in series query:", err);
        // Return empty array on timeout
        return [];
      });

    // OPTIMIZATION: Simplified processing
    const processedSeries = series.map(item => ({
      ...item,
      totalReads: item.reads || 0,
      episodeCount: item.episodeCount || 0,
      likesCount: Array.isArray(item.likes) ? item.likes.length : 0,
      type: "series"
    }));

    console.timeEnd("fetchAllSeries");

    // If no series found, return a default series with placeholder data
    if (processedSeries.length === 0) {
      return res.status(200).json({
        success: true,
        series: [{
          _id: "placeholder-1",
          title: "Sample Series",
          coverImage: "/placeholder-cover.jpg",
          summary: "This is a placeholder series while we load your content.",
          author: { _id: "author-1", username: "NovellHub" },
          totalReads: 0,
          episodeCount: 0,
          likesCount: 0,
          type: "series",
          isPlaceholder: true
        }],
        total: 1,
        message: "Showing placeholder content while loading"
      });
    }

    return res.status(200).json({
      success: true,
      series: processedSeries,
      total: processedSeries.length
    });
  } catch (error) {
    console.error("Error fetching all series:", error);
    return res.status(200).json({
      success: true,
      series: [{
        _id: "placeholder-1",
        title: "Sample Series",
        coverImage: "/placeholder-cover.jpg",
        summary: "This is a placeholder series while we load your content.",
        author: { _id: "author-1", username: "NovellHub" },
        totalReads: 0,
        episodeCount: 0,
        likesCount: 0,
        type: "series",
        isPlaceholder: true
      }],
      total: 1,
      message: "Error occurred, showing placeholder content"
    });
  }
};



// Optimize the getTrendingSeries function
const getTrendingSeries = async (req, res, callback) => {
  try {
    const { language, sortBy = "reads", sortOrder = "desc", limit = 12 } = req.query;
    console.time("fetchTrendingSeries");

    // Create a base query
    const query = { status: "published" }; // Only get published series
    if (language && language !== "ALL") query.language = language;

    // Determine sort options
    let sortOptions = {};
    if (sortBy === "reads") {
      sortOptions = { reads: sortOrder === "desc" ? -1 : 1 };
    } else if (sortBy === "newest") {
      sortOptions = { createdAt: sortOrder === "desc" ? -1 : 1 };
    } else if (sortBy === "rating") {
      sortOptions = { averageRating: sortOrder === "desc" ? -1 : 1 };
    } else {
      // Default to reads
      sortOptions = { reads: -1 };
    }

    // OPTIMIZATION: Only select necessary fields and use lean() for better performance
    const series = await Series.find(query)
      .select("title coverImage summary author categories likes reads createdAt episodeCount averageRating language")
      .populate("author", "username")
      .sort(sortOptions)
      .limit(parseInt(limit))
      .lean()
      .maxTimeMS(10000); // Set a 10-second timeout for the query

    console.timeEnd("fetchTrendingSeries");

    // Process and return the data
    const processedSeries = series.map((item) => ({
      ...item,
      totalReads: item.reads || 0,
      episodeCount: item.episodeCount || 0,
      likesCount: Array.isArray(item.likes) ? item.likes.length : 0,
      type: "series",
    }));

    // If no series found, return a default series with placeholder data
    if (processedSeries.length === 0) {
      const response = {
        success: true,
        series: [{
          _id: "placeholder-1",
          title: "Trending Series",
          coverImage: "/placeholder-cover.jpg",
          summary: "This is a placeholder trending series while we load your content.",
          author: { _id: "author-1", username: "NovellHub" },
          totalReads: 0,
          episodeCount: 0,
          likesCount: 0,
          type: "series",
          isPlaceholder: true
        }],
        total: 1,
        message: "Showing placeholder content while loading"
      };
      
      if (callback) callback(response);
      return res.status(200).json(response);
    }

    const response = {
      success: true,
      series: processedSeries,
      total: processedSeries.length
    };
    
    if (callback) callback(response);
    return res.status(200).json(response);
  } catch (error) {
    console.error("Error fetching trending series:", error);
    const response = {
      success: true,
      series: [{
        _id: "placeholder-1",
        title: "Trending Series",
        coverImage: "/placeholder-cover.jpg",
        summary: "This is a placeholder trending series while we load your content.",
        author: { _id: "author-1", username: "NovellHub" },
        totalReads: 0,
        episodeCount: 0,
        likesCount: 0,
        type: "series",
        isPlaceholder: true
      }],
      total: 1,
      message: "Error occurred, showing placeholder content"
    };
    
    if (callback) callback(response);
    return res.status(200).json(response);
  }
};





// Add this function to the controller
const getAllSeriesByLanguage = async (req, res) => {
  try {
    console.log("Processing getAllSeriesByLanguage with query:", req.query);
    const { 
      sortBy = "popular", 
      sortOrder = "desc",
      category = null,
      search = "",
      language = null,
      page = 1,
      limit = 12
    } = req.query;

    // Create base query
    const query = { 
      status: "published"
    };
    
    // Add language filter based on the user's selected language
    if (language && language !== "ALL" && language !== "English") {
      console.log(`Filtering by language: ${language}`);
      query.language = language;
    }
    
    // Add category filter if provided
    if (category) {
      query.categories = { $in: [category] };
    }

    // Add search functionality
    if (search && search.trim() !== "") {
      const searchRegex = new RegExp(search, "i");
      query.$or = [
        { title: searchRegex },
        { summary: searchRegex }
      ];
    }
    
    console.log("Final query:", JSON.stringify(query));
    
    // Determine sort options
    let sortOptions = {};
    if (sortBy === "popular") {
      sortOptions = { reads: sortOrder === "desc" ? -1 : 1 };
    } else if (sortBy === "newest") {
      sortOptions = { createdAt: sortOrder === "desc" ? -1 : 1 };
    } else if (sortBy === "rating") {
      sortOptions = { averageRating: sortOrder === "desc" ? -1 : 1 };
    } else {
      // Default to popularity
      sortOptions = { reads: -1 };
    }
    
    // Add secondary sort by createdAt to ensure consistent ordering
    sortOptions.createdAt = sortOrder === "desc" ? -1 : 1;

    // Calculate pagination
    const skip = (parseInt(page) - 1) * parseInt(limit);
    
    // First, check if we have any series with this language
    const count = await Series.countDocuments(query);
    console.log(`Found ${count} series matching the query`);

    // If no series found with the exact language, try a more flexible approach
    if (count === 0 && language && language !== "ALL" && language !== "English") {
      console.log("No series found with exact language match, trying case-insensitive search");
      
      // Remove the language filter and use a regex instead
      delete query.language;
      
      // Try a case-insensitive search for the language
      if (language) {
        const languageRegex = new RegExp(language, "i");
        query.language = languageRegex;
      }
      
      console.log("Updated query:", JSON.stringify(query));
    }
    
    // Use Promise.all to run queries in parallel for better performance
    const [series, total] = await Promise.all([
      Series.find(query)
        .select("title coverImage summary author categories likes reads createdAt episodes averageRating reviewCount episodeCount language")
        .populate("author", "username profileImage")
        .sort(sortOptions)
       .skip(skip)
       .limit(parseInt(limit))
       .lean(),
     Series.countDocuments(query)
   ]);

   console.log(`Retrieved ${series.length} series from database`);

   // If still no series found, return all series as a fallback
   if (series.length === 0) {
     console.log("No series found with language filter, returning all published series as fallback");
     
     const allSeries = await Series.find({ status: "published" })
       .select("title coverImage summary author categories likes reads createdAt episodes averageRating reviewCount episodeCount language")
       .populate("author", "username profileImage")
       .sort(sortOptions)
       .limit(parseInt(limit))
       .lean();
     
     console.log(`Fallback retrieved ${allSeries.length} series`);
     
     // Process and return the fallback series
     const processedFallbackSeries = allSeries.map(item => ({
       ...item,
       totalReads: item.reads || 0,
       episodeCount: Array.isArray(item.episodes) ? item.episodes.length : 0,
       likesCount: Array.isArray(item.likes) ? item.likes.length : 0,
       type: "series"
     }));
     
     return res.status(200).json({
       success: true,
       series: processedFallbackSeries,
       total: processedFallbackSeries.length,
       pagination: {
         total: allSeries.length,
         page: parseInt(page),
         pages: Math.ceil(allSeries.length / parseInt(limit)),
         hasMore: allSeries.length > parseInt(limit)
       },
       message: "No series found with the specified language, showing all series instead"
     });
   }

   // Process series data to include calculated fields
   const processedSeries = series.map(item => {
     // Calculate total reads if not already present
     const totalReads = item.reads || 
       (item.episodes?.reduce((sum, ep) => sum + (ep.reads || 0), 0) || 0);
     
     // Calculate episode count if not already present
     const episodeCount = item.episodeCount || 
       (Array.isArray(item.episodes) ? item.episodes.length : 0);
     
     // Calculate likes count
     const likesCount = Array.isArray(item.likes) ? item.likes.length : 0;
     
     return {
       ...item,
       totalReads,
       episodeCount,
       likesCount,
       type: "series"
     };
   });

   // Return success response with pagination info
   return res.status(200).json({
     success: true,
     series: processedSeries,
     total: total,
     pagination: {
       total,
       page: parseInt(page),
       pages: Math.ceil(total / parseInt(limit)),
       hasMore: skip + processedSeries.length < total
     }
   });
 } catch (error) {
   console.error("Error fetching series by language:", error);
   
   // Return error response
   return res.status(500).json({
     success: false,
     error: "An internal server error occurred while fetching series.",
     details: process.env.NODE_ENV === "development" ? error.message : undefined
   });
 }
};








const followUser = async (req, res) => {
  try {
    console.log(
      "Series controller: Follow user request received for:",
      req.params.userId
    );

    // Find users using findById to get the full documents
    const userToFollow = await User.findById(req.params.userId);
    const currentUser = await User.findById(req.userId);

    if (!userToFollow || !currentUser) {
      console.log(
        "User not found:",
        !userToFollow ? "target user" : "current user"
      );
      return res.status(404).json({ error: "User not found" });
    }

    if (userToFollow._id.toString() === currentUser._id.toString()) {
      console.log("User attempted to follow themselves");
      return res.status(400).json({ error: "Cannot follow yourself" });
    }

    // Check if already following
    const isFollowing = userToFollow.followers.some(
      (id) => id.toString() === req.userId
    );

    console.log(
      "Current follow status:",
      isFollowing ? "following" : "not following"
    );

    // Use direct MongoDB operations to update the follow status
    // This bypasses Mongoose validation issues
    if (isFollowing) {
      // Unfollow - pull the IDs from the arrays
      await User.updateOne(
        { _id: userToFollow._id },
        { $pull: { followers: req.userId } }
      );

      await User.updateOne(
        { _id: req.userId },
        { $pull: { following: userToFollow._id } }
      );

      console.log("Unfollow operation completed");
    } else {
      // Follow - add the IDs to the arrays
      await User.updateOne(
        { _id: userToFollow._id },
        { $addToSet: { followers: req.userId } }
      );

      await User.updateOne(
        { _id: req.userId },
        { $addToSet: { following: userToFollow._id } }
      );

      console.log("Follow operation completed");

      // Try to add notification, but don't let it fail the whole operation
      try {
        // Use updateOne to add notification directly
        await User.updateOne(
          { _id: userToFollow._id },
          {
            $push: {
              notifications: {
                type: "follow",
                message: `${currentUser.username} started following you`,
                fromUser: currentUser._id,
                contentId: currentUser._id,
                contentModel: "User",
                createdAt: new Date(),
                isRead: false,
              },
            },
          }
        );
        console.log("Notification added successfully");
      } catch (notifError) {
        // Log the error but continue
        console.error(
          "Failed to add notification, but follow succeeded:",
          notifError
        );
      }
    }

    // Verify the operation worked by checking the updated status
    const updatedUserToFollow = await User.findById(req.params.userId);
    const newIsFollowing = updatedUserToFollow.followers.some(
      (id) => id.toString() === req.userId
    );

    console.log(
      "New follow status:",
      newIsFollowing ? "following" : "not following"
    );

    res.status(200).json({
      success: true,
      isFollowing: newIsFollowing,
      message: isFollowing
        ? "Unfollowed successfully"
        : "Following successfully",
    });
  } catch (error) {
    console.error("Follow user error:", error);
    res.status(500).json({ error: error.message });
  }
};

// Check the getSeriesByCategory function
const getSeriesByCategory = async (req, res) => {
  try {
    const { category } = req.params;
    const series = await Series.find({
      categories: category,
      status: { $ne: "draft" },
    })
      .populate("author", "username profileImage")
      .populate("episodes")
      .populate("likes") // Make sure likes are populated
      .sort({ createdAt: -1 });

    res.status(200).json({
      success: true,
      series,
      message: "Series by category fetched successfully",
    });
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};

const updateSeriesDetails = async (req, res) => {
  try {
    const { title, categories, coverImage, storyType, summary, language } =
      req.body;
    const series = await Series.findOne({
      _id: req.params.id,
      author: req.userId,
    });

    if (!series) return res.status(404).json({ error: "Series not found" });

    if (title) series.title = title;
    if (categories) series.categories = categories;
    if (coverImage) series.coverImage = coverImage;
    if (storyType) series.storyType = storyType;
    if (summary) series.summary = summary;
    if (language) series.language = language;
    await series.save();
    res.status(200).json({
      success: true,
      message: "Series details updated successfully",
    });
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};

const uploadCoverImage = async (req, res) => {
  try {
    if (!req.file) {
      return res.status(400).json({ error: "No file uploaded" });
    }

    const imageUrl = `/uploads/${req.file.filename}`;

    const series = await Series.findByIdAndUpdate(
      req.params.id,
      { coverImage: imageUrl },
      { new: true }
    );

    if (!series) {
      return res.status(404).json({ error: "Series not found" });
    }

    res.status(200).json({
      success: true,
      coverImage: imageUrl,
      message: "Cover image updated successfully",
    });
  } catch (error) {
    console.error("Upload error:", error); // Debug log
    res.status(500).json({ error: error.message });
  }
};
const addEpisodeReader = async (req, res) => {
  try {
    const series = await Series.findById(req.params.id);
    if (!series) {
      return res.status(404).json({ error: "Series not found" });
    }

    const episode = series.episodes.id(req.params.episodeId);
    if (!episode) {
      return res.status(404).json({ error: "Episode not found" });
    }

    // Initialize readers array if it doesn't exist
    if (!episode.readers) {
      episode.readers = [];
    }

    // Check if reader exists and add if new
    const readerExists = episode.readers.some(
      (readerId) => readerId.toString() === req.userId
    );

    if (!readerExists) {
      episode.readers.push(req.userId);
      episode.totalReaders = episode.readers.length;

      // Use findOneAndUpdate for atomic operation
      await Series.findOneAndUpdate(
        { _id: req.params.id },
        { $set: { episodes: series.episodes } },
        { new: true }
      );
    }

    res.status(200).json({
      success: true,
      totalReaders: episode.readers.length,
      message: readerExists
        ? "Reader already counted"
        : "Reader added successfully",
    });
  } catch (error) {
    console.error("Error in addEpisodeReader:", error);
    res.status(500).json({
      error: "Failed to update reader count",
      details: error.message,
    });
  }
};


// In Series_Controller.js

const checkEpisodeLock = async (req, res) => {
  try {
    const { id, episodeId } = req.params;
    const userId = req.userId;

    console.log(
      `Checking lock status for user ${userId}, series ${id}, episode ${episodeId}`
    );

    // Find the series
    const series = await Series.findById(id);
    if (!series) {
      return res.status(404).json({ error: "Series not found" });
    }

    // Check if user is the author
    const isAuthor = series.author.toString() === userId;

    // If user is the author, episode is always unlocked
    if (isAuthor) {
      console.log(`User ${userId} is the author, episode is unlocked`);
      return res.json({ isLocked: false });
    }

    // Find the episode index
    const episodeIndex = series.episodes.findIndex(
      (ep) => ep._id.toString() === episodeId
    );
    if (episodeIndex === -1) {
      return res.status(404).json({ error: "Episode not found" });
    }

    // First 4 episodes are always free
    if (episodeIndex < 4) {
      console.log(
        `Episode ${episodeId} is one of the first 4 episodes, so it's free`
      );
      return res.json({ isLocked: false });
    }

    // Get user to check unlocked episodes
    const user = await User.findById(userId);
    if (!user) {
      return res.status(404).json({ error: "User not found" });
    }

    // Convert series ID to string to ensure consistent key format
    const seriesIdStr = series._id.toString();

    // Initialize unlockedEpisodes if it doesn't exist
    if (!user.unlockedEpisodes) {
      user.unlockedEpisodes = {};
    }

    // Check if episode is unlocked for this user
    const seriesUnlocked = user.unlockedEpisodes[seriesIdStr] || [];
    const isUnlocked = seriesUnlocked.includes(episodeId);

    console.log(
      `User's unlocked episodes for series ${seriesIdStr}:`,
      seriesUnlocked.length > 0 ? seriesUnlocked : "none"
    );
    console.log(
      `Is episode ${episodeId} unlocked for user ${userId}? ${isUnlocked}`
    );

    res.json({ isLocked: !isUnlocked });
  } catch (error) {
    console.error("Error checking episode lock status:", error);
    res.status(500).json({ error: error.message });
  }
};

// Add this function to check if a specific episode is locked
const getEpisodeLockStatus = async (req, res) => {
  try {
    const { id, episodeId } = req.params;
    const userId = req.userId;

    // Find the series
    const series = await Series.findById(id);
    if (!series) {
      return res.status(404).json({ error: "Series not found" });
    }

    // Check if user is the author
    const isAuthor = series.author.toString() === userId;

    // If user is the author, episode is always unlocked
    if (isAuthor) {
      return res.json({ isLocked: false, isAuthor: true });
    }

    // Find the episode index
    const episodeIndex = series.episodes.findIndex(
      (ep) => ep._id.toString() === episodeId
    );
    if (episodeIndex === -1) {
      return res.status(404).json({ error: "Episode not found" });
    }

    // First 4 episodes are always free
    if (episodeIndex < 4) {
      return res.json({ isLocked: false, isFree: true });
    }

    // Get user to check unlocked episodes
    const user = await User.findById(userId);
    if (!user) {
      return res.status(404).json({ error: "User not found" });
    }

    // Check if episode is unlocked for this user
    const seriesIdStr = series._id.toString();
    if (!user.unlockedEpisodes) {
      user.unlockedEpisodes = {};
    }
    
    const seriesUnlocked = user.unlockedEpisodes[seriesIdStr] || [];
    const isUnlocked = seriesUnlocked.includes(episodeId);

    return res.json({ 
      isLocked: !isUnlocked,
      episodeIndex,
      isAuthor: false,
      isFree: false
    });
  } catch (error) {
    console.error("Error checking episode lock status:", error);
    res.status(500).json({ error: error.message });
  }
};



const checkSeriesEpisodesLockStatus = async (req, res) => {
  try {
    const { id } = req.params;
    const userId = req.userId;
    
    console.log(`Checking lock status for all episodes in series ${id} for user ${userId}`);
    
    // Find the series
    const series = await Series.findById(id);
    if (!series) {
      return res.status(404).json({ error: "Series not found" });
    }
    
    // Check if user is the author
    const isAuthor = series.author.toString() === userId;
    
    // If user is the author, all episodes are unlocked
    if (isAuthor) {
      const unlockStatus = {};
      series.episodes.forEach(ep => {
        unlockStatus[ep._id.toString()] = false; // false means unlocked
      });
      
      console.log("User is author, all episodes unlocked:", unlockStatus);
      return res.json({ 
        isAuthor: true,
        lockStatus: unlockStatus
      });
    }
    
    // Get user to check unlocked episodes
    const user = await User.findById(userId);
    if (!user) {
      return res.status(404).json({ error: "User not found" });
    }
    
    // Convert series ID to string to ensure consistent key format
    const seriesIdStr = series._id.toString();
    
    // Initialize unlockedEpisodes if it doesn't exist
    if (!user.unlockedEpisodes) {
      user.unlockedEpisodes = {};
    }
    
    // Get the user's unlocked episodes for this series
    const seriesUnlocked = user.unlockedEpisodes[seriesIdStr] || [];
    
    console.log(`User's unlocked episodes for series ${seriesIdStr}:`, seriesUnlocked);
    
    // Create a map of episode IDs to lock status
    const lockStatus = {};
    
    series.episodes.forEach((episode, index) => {
      const episodeId = episode._id.toString();
      // First 4 episodes are always free
      if (index < 4) {
        lockStatus[episodeId] = false; // unlocked
      } else {
        // Check if episode is in the unlocked list
        const isUnlocked = seriesUnlocked.includes(episodeId);
        lockStatus[episodeId] = !isUnlocked; // true means locked
      }
    });
    
    console.log("Final lock status for all episodes:", lockStatus);
    
    res.json({
      isAuthor: false,
      lockStatus: lockStatus
    });
  } catch (error) {
    console.error("Error checking series episodes lock status:", error);
    res.status(500).json({ error: error.message });
  }
};


const unlockEpisode = async (req, res) => {
  try {
    const { id, episodeId } = req.params;
    const userId = req.userId;
    
    console.log(`Attempting to unlock episode ${episodeId} in series ${id} for user ${userId}`);
    
    // Find the series and episode
    const series = await Series.findById(id);
    if (!series) {
      return res.status(404).json({ error: "Series not found" });
    }
    
    const episode = series.episodes.id(episodeId);
    if (!episode) {
      return res.status(404).json({ error: "Episode not found" });
    }
    
    // Get the user
    const user = await User.findById(userId);
    if (!user) {
      return res.status(404).json({ error: "User not found" });
    }
    
    // Convert series ID to string to ensure consistent key format
    const seriesIdStr = series._id.toString();
    
    // Initialize unlockedEpisodes if it doesn't exist
    if (!user.unlockedEpisodes) {
      user.unlockedEpisodes = {};
    }
    
    // Initialize array for this series if it doesn't exist
    if (!user.unlockedEpisodes[seriesIdStr]) {
      user.unlockedEpisodes[seriesIdStr] = [];
    }
    
    // Check if user already unlocked this episode
    const isAlreadyUnlocked = user.unlockedEpisodes[seriesIdStr].includes(episodeId);
    
    if (isAlreadyUnlocked) {
      console.log(`Episode ${episodeId} already unlocked for user ${userId}`);
      return res.status(200).json({ 
        success: true, 
        message: "Episode already unlocked", 
        remainingCoins: user.coins,
        earnedCoins: user.earnedCoins || 0,
        totalCoins: (user.coins || 0) + (user.earnedCoins || 0),
        isAlreadyUnlocked: true
      });
    }
    
    // Check if user has enough coins
    if ((user.coins || 0) < 4) {
      return res.status(400).json({ error: "Not enough coins to unlock this episode" });
    }
    
    // Add episode to unlocked list
    user.unlockedEpisodes[seriesIdStr].push(episodeId);
    
    console.log(`Added episode ${episodeId} to unlocked list for user ${userId}`);
    console.log(`Updated unlocked episodes:`, user.unlockedEpisodes);
    
    // Define the coin cost for unlocking an episode
    const coinCost = 4;
    
    // Deduct coins from user
    user.coins = (user.coins || 0) - coinCost;
    
    // Create transaction record with detailed information
    const transaction = {
      amount: coinCost,
      from: userId,
      fromUsername: user.username,
      seriesId: series._id,
      episodeId: episodeId,
      episodeTitle: episode.title,
      storyTitle: series.title,
      transactionType: "purchase",
      transactionDate: new Date()
    };
    
    // Add transaction to user's records
    if (!user.coinTransactions) {
      user.coinTransactions = [];
    }
    user.coinTransactions.push(transaction);
    
    // Find the author and add coins
    const author = await User.findById(series.author);
    if (author) {
      // Add coins to author's earned coins
      author.earnedCoins = (author.earnedCoins || 0) + coinCost;
      
      // Add transaction to author's records
      if (!author.coinTransactions) {
        author.coinTransactions = [];
      }
      
      const authorTransaction = {
        ...transaction,
        to: author._id
      };
      
      author.coinTransactions.push(authorTransaction);
      
      if (coinCost > 0) {
        // Update coins spent for reader level
        await updateReaderLevel(userId, 4);
      }

      // Create a notification for the author
      const notification = {
        type: "purchase",
        message: `${user.username} unlocked episode "${episode.title}" in your series "${series.title}"`,
        fromUser: userId,
        contentId: series._id,
        contentModel: "Series",
        isRead: false,
        createdAt: new Date()
      };
      
      // Update author
      await User.findByIdAndUpdate(author._id, {
        $inc: { earnedCoins: coinCost },
        $push: {
          coinTransactions: authorTransaction,
          notifications: notification
        }
      });
    }
    
    // Update episode stats
    episode.unlocks = (episode.unlocks || 0) + 1;
    episode.coinEarnings = (episode.coinEarnings || 0) + coinCost;
    
    // Save changes to series
    await series.save();
    
    // Save user changes - use a direct update to ensure it's saved correctly
    await User.findByIdAndUpdate(userId, {
      $set: {
        coins: user.coins,
        unlockedEpisodes: user.unlockedEpisodes
      },
      $push: {
        coinTransactions: transaction
      }
    });
    
    console.log(`User ${userId} successfully unlocked episode ${episodeId} in series ${id}`);
    
    res.status(200).json({
      success: true,
      message: "Episode unlocked successfully",
      remainingCoins: user.coins,
      earnedCoins: user.earnedCoins || 0,
      totalCoins: (user.coins || 0) + (user.earnedCoins || 0)
    });
  } catch (error) {
    console.error("Error unlocking episode:", error);
    res.status(500).json({ error: error.message });
  }
};




// Update getCoinEarnings function
const getCoinEarnings = async (req, res) => {
  try {
    const user = await User.findById(req.userId);
    if (!user) {
      return res.status(404).json({ error: "User not found" });
    }

    // Calculate total earnings
    const totalEarnings = user.coinTransactions
      ? user.coinTransactions.reduce(
          (total, transaction) => total + transaction.amount,
          0
        )
      : 0;

    res.status(200).json({
      success: true,
      coins: user.coins || 0,
      earnedCoins: user.earnedCoins || 0,
      totalCoins: (user.coins || 0) + (user.earnedCoins || 0),
      transactions: user.coinTransactions || [],
      totalEarnings,
      message: "Coin earnings retrieved successfully",
    });
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};

// Get detailed coin earnings for a user
const getDetailedCoinEarnings = async (req, res) => {
  try {
    const userId = req.userId;

    // Find the user and populate the transaction data
    const user = await User.findById(userId)
      .populate({
        path: "coinTransactions.from",
        select: "username profileImage",
      })
      .populate({
        path: "coinTransactions.seriesId",
        select: "title",
      });

    if (!user) {
      return res.status(404).json({ error: "User not found" });
    }

    // Format the transactions with additional details
    const transactions = user.coinTransactions.map((tx) => {
      return {
        amount: tx.amount,
        from: tx.from?._id || null,
        fromUsername: tx.from?.username || "Anonymous Reader",
        seriesId: tx.seriesId?._id || null,
        seriesTitle: tx.seriesId?.title || null,
        episodeId: tx.episodeId || null,
        episodeTitle: tx.episodeTitle || null,
        transactionDate: tx.transactionDate,
      };
    });

    // Sort transactions by date (newest first)
    transactions.sort(
      (a, b) => new Date(b.transactionDate) - new Date(a.transactionDate)
    );

    res.status(200).json({
      success: true,
      transactions,
      message: "Detailed coin earnings retrieved successfully",
    });
  } catch (error) {
    console.error("Error getting detailed coin earnings:", error);
    res.status(500).json({ error: error.message });
  }
};

// Add a new function to update series status
const updateSeriesStatus = async (req, res) => {
  try {
    const { status } = req.body;

    // Validate status value
    if (!status || !["ongoing", "completed"].includes(status)) {
      return res
        .status(400)
        .json({ error: "Invalid status. Must be 'ongoing' or 'completed'" });
    }

    const series = await Series.findOne({
      _id: req.params.id,
      author: req.userId,
    });

    if (!series) {
      return res.status(404).json({ error: "Series not found" });
    }

    // Update the status
    series.status = status;
    await series.save();

    res.status(200).json({
      success: true,
      message: "Series status updated successfully",
      status: series.status,
    });
  } catch (error) {
    console.error("Error updating series status:", error);
    res.status(500).json({ error: error.message });
  }
};

// Add this to your seriesController.js file
const getBasicEpisodeInfo = async (req, res) => {
  try {
    const { id, episodeId } = req.params;
    
    // Find the series
    const series = await Series.findById(id)
      .select('title episodes author')
      .populate('author', 'username');
    
    if (!series) {
      return res.status(404).json({ success: false, error: "Series not found" });
    }
    
    // Find the episode in the series
    const episodeIndex = series.episodes.findIndex(ep => ep._id.toString() === episodeId);
    
    if (episodeIndex === -1) {
      return res.status(404).json({ success: false, error: "Episode not found in this series" });
    }
    
    // Get the episode
    const episode = series.episodes[episodeIndex];
    
    if (!episode) {
      return res.status(404).json({ success: false, error: "Episode not found" });
    }
    
    // Return basic info without content
    return res.json({
      success: true,
      episode: {
        _id: episode._id,
        title: episode.title,
        createdAt: episode.createdAt,
        updatedAt: episode.updatedAt,
        likes: episode.likes || [],
        totalReaders: episode.totalReaders || 0
      },
      series: {
        _id: series._id,
        title: series.title,
        author: series.author
      },
      episodeIndex
    });
  } catch (error) {
    console.error("Error fetching basic episode info:", error);
    return res.status(500).json({ success: false, error: "Server error" });
  }
};





module.exports = {
  // Base Series Functions
  createSeries,
  getUserSeries,
  saveDraft,
  getDrafts,
  deleteSeries,
  deleteSeriesDraft,
  getSeriesById,

  // Episode Functions
  addEpisode,
  getEpisodeById,
  updateEpisode,
  deleteEpisode,

  // Series Interaction Functions
  likeSeries,
  incrementReads,

  // Episode Interaction Functions
  likeEpisode,
  incrementEpisodeReads,
  likeEpisodeReply,
  updateEpisodeReply,
  deleteEpisodeReply,
  
  // Review Functions
  addEpisodeReview,
  getEpisodeReviews,
  addEpisodeReviewReply,
  likeEpisodeReview,
  updateEpisodeReview,
  deleteEpisodeReview,

  // Series Listing Functions
  getAllPublishedSeries,
  getAllSeriesByLanguage, // Add the new function here
  getTrendingSeries,
  followUser,
  getSeriesByCategory,
  updateSeriesDetails,
  uploadCoverImage,
  addEpisodeReader,
  checkSeriesEpisodesLockStatus,
  checkEpisodeLock,
  getEpisodeLockStatus,
  unlockEpisode,
  getCoinEarnings,
  getDetailedCoinEarnings,
  updateSeriesStatus,
  getBasicEpisodeInfo
};
