const Story = require("../Model/Story_Model");
const User = require("../Model/User_Model"); // Add this import at the top
const { createNotification } = require("../utils/notifications");

const calculateAverageRating = (reviews) => {
  if (!reviews.length) return 0;
  const sum = reviews.reduce((acc, review) => acc + review.rating, 0);
  return (sum / reviews.length).toFixed(1);
};

const createStory = async (req, res) => {
  try {
    const {
      title,
      content,
      coverImage,
      summary,
      storyType,
      categories,
      status,
      language,
    } = req.body;
    const user = await User.findById(req.userId);

    const story = new Story({
      title,
      content,
      coverImage,
      summary,
      author: req.userId,
      storyType: storyType || "Fiction",
      categories,
      status: status || "draft",
      language: language || "EN",
    });

    const savedStory = await story.save();

    // Update word count stats
    if (status === "published") {
      const wordCount = content
        .split(/\s+/)
        .filter((word) => word.length > 0).length;
      const dateKey = new Date().toISOString().split("T")[0];

      await User.findByIdAndUpdate(req.userId, {
        $inc: {
          "writingStats.totalWordCount": wordCount,
          "writingStats.totalEarnings": wordCount * 0.10,
          [`writingStats.dailyWordCount.${dateKey}.count`]: wordCount,
          [`writingStats.dailyWordCount.${dateKey}.earnings`]: wordCount * 0.10,
        },
      });
    }

    res.status(201).json({
      success: true,
      story: savedStory,
    });
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};

const getUserStories = async (req, res) => {
  try {
    const stories = await Story.find({ author: req.userId })
      .populate("author", "username")
      .populate("readHistory.user", "username")
      .sort({ createdAt: -1 });

    res.status(200).json({
      success: true,
      stories,
      message: "Stories fetched successfully",
    });
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};

const updateStory = async (req, res) => {
  try {
    const oldStory = await Story.findById(req.params.id);
    if (!oldStory) return res.status(404).json({ error: "Story not found" });

    // Calculate word count difference for stats
    const oldWordCount = oldStory.content
      .split(/\s+/)
      .filter((word) => word.length > 0).length;
    const newWordCount = req.body.content
      .split(/\s+/)
      .filter((word) => word.length > 0).length;
    const wordCountDiff = newWordCount - oldWordCount;

    // Update user writing stats if word count changed
    if (wordCountDiff !== 0) {
      const today = new Date().toISOString().split("T")[0];
      await User.findByIdAndUpdate(req.userId, {
        $inc: {
          "writingStats.totalWordCount": wordCountDiff,
          "writingStats.totalEarnings": wordCountDiff * 0.10,
          [`writingStats.dailyWordCount.${today}.count`]: wordCountDiff,
          [`writingStats.dailyWordCount.${today}.earnings`]: wordCountDiff * 0.10,
        },
      });
    }

    // Update all story fields
    oldStory.title = req.body.title || oldStory.title;
    oldStory.content = req.body.content;
    oldStory.coverImage = req.body.coverImage || oldStory.coverImage;
    oldStory.summary = req.body.summary || oldStory.summary;
    oldStory.storyType = req.body.storyType || oldStory.storyType;
    oldStory.categories = req.body.categories || oldStory.categories;
    oldStory.status = req.body.status || oldStory.status;
    oldStory.language = req.body.language || oldStory.language;

    const updatedStory = await oldStory.save();

    res.status(200).json({
      success: true,
      message: "Story updated successfully",
      story: updatedStory
    });
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};


const deleteStory = async (req, res) => {
  try {
    const story = await Story.findById(req.params.id);
    const wordCount = story.content
      .split(/\s+/)
      .filter((word) => word.length > 0).length;

    const dateKey = story.createdAt.toISOString().split("T")[0];

    // Update user's writing stats
    await User.findByIdAndUpdate(req.userId, {
      $inc: {
        "writingStats.totalWordCount": -wordCount,
        "writingStats.earnings": -(wordCount * 0.10),
        [`writingStats.dailyWordCount.${dateKey}.count`]: -wordCount,
        [`writingStats.dailyWordCount.${dateKey}.earnings`]: -(
          wordCount * 0.07
        ),
      },
      $pull: {
        "writingStats.contentWordCounts": { contentId: story._id },
      },
    });

    // Delete story and notifications
    await Promise.all([
      Story.findByIdAndDelete(req.params.id),
      User.updateMany(
        { "notifications.contentId": story._id },
        { $pull: { notifications: { contentId: story._id } } }
      ),
    ]);

    res.status(200).json({
      success: true,
      message: "Story and related data deleted successfully",
    });
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};

const deleteStoryDraft = async (req, res) => {
  try {
    const draft = await Story.findOneAndDelete({
      _id: req.params.id,
      author: req.userId,
      status: "draft",
    });
    if (!draft) return res.status(404).json({ error: "Draft not found" });
    res.status(200).json({
      success: true,
      message: "Story Draft deleted successfully",
    });
  } catch (error) {
    res.status(500).json({
      error: error.message,
    });
  }
};

const getStoryDrafts = async (req, res) => {
  try {
    const drafts = await Story.find({
      author: req.userId,
      status: "draft",
    }).sort({ createdAt: -1 });
    res.status(200).json({
      success: true,
      drafts,
      message: "Story Drafts Fetched Successfully",
    });
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};

const getStoryById = async (req, res) => {
  try {
    const story = await Story.findById(req.params.id)
      .populate({
        path: "author",
        select: "username profileImage followers",
        populate: {
          path: "followers",
          select: "_id",
        },
      })
      .populate("reviews.user", "username")
      .populate("reviews.replies.user", "username")
      .populate("reviews.replies.mentionedUser", "username")
      .populate("likes", "username")
      .populate("readHistory.user", "username");

    if (!story) return res.status(404).json({ error: "Story not found" });

    // Add isLiked flag
    const isLiked = story.likes.some(like => like._id.toString() === req.userId);

    res.status(200).json({
      success: true,
      story,
      isLiked,
      message: "Story Fetched Successfully",
    });
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};


const likeStory = async (req, res) => {
  try {
    const story = await Story.findById(req.params.id);
    if (!story) return res.status(404).json({ error: "Story not found" });

    // Initialize likes array if it doesn't exist
    if (!Array.isArray(story.likes)) {
      story.likes = [];
    }

    const likeIndex = story.likes.indexOf(req.userId);
    if (likeIndex === -1) {
      story.likes.push(req.userId);

      const user = await User.findById(req.userId);
      await createNotification(story.author, {
        type: "like",
        message: `${user.username} liked your story "${story.title}"`,
        fromUser: req.userId,
        contentId: story._id,
        contentModel: "Story",
      });
    } else {
      story.likes.splice(likeIndex, 1);
    }
    await story.save();

    res.json({
      success: true,
      likes: story.likes,
      isLiked: likeIndex === -1,
    });
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};

// Optimize getAllStories to load all data at once and randomize results
const getAllStories = async (req, res) => {
  try {
    const { 
      language, 
      sortBy = "reads", 
      sortOrder = "desc", 
      search = "", 
      category = null,
      randomize = false // Add a new parameter to control randomization
    } = req.query;
    
    console.time("fetchAllStories");

    // Create a base query
    const query = { status: { $ne: "draft" } };
    
    // Add language filter if specified, but make it case-insensitive
    if (language && language !== "ALL") {
      // Use a regex for case-insensitive matching
      const languageRegex = new RegExp(language, "i");
      query.language = languageRegex;
    }
    
    // Add category filter if provided
    if (category) {
      query.categories = { $in: [category] };
    }

    // Add search functionality
    if (search && search.trim() !== "") {
      const searchRegex = new RegExp(search, "i");
      query.$or = [
        { title: searchRegex },
        { content: searchRegex }
      ];
    }

    console.log("Stories query:", JSON.stringify(query));

    // Determine sort options
    let sortOptions = {};
    
    // If randomize is true, we'll skip sorting here and randomize later
    if (!randomize) {
      if (sortBy === "reads" || sortBy === "popular") {
        sortOptions = { reads: sortOrder === "desc" ? -1 : 1 };
      } else if (sortBy === "newest") {
        sortOptions = { createdAt: sortOrder === "desc" ? -1 : 1 };
      } else if (sortBy === "rating") {
        sortOptions = { averageRating: sortOrder === "desc" ? -1 : 1 };
      } else {
        // Default to reads
        sortOptions = { reads: -1 };
      }
      
      // Add secondary sort for consistent ordering
      sortOptions.createdAt = -1;
    }

    console.log(`Sorting options: ${JSON.stringify(sortOptions)}`);

    // Use Promise.all to run queries in parallel for better performance
    const [stories, total] = await Promise.all([
      Story.find(query)
        .select("title coverImage content author categories likes reads createdAt averageRating reviewCount language")
        .populate("author", "username profileImage")
        .sort(randomize ? {} : sortOptions) // Only apply sorting if not randomizing
        .lean(),
      Story.countDocuments(query)
    ]);

    // Process the data
    let processedStories = stories.map((item) => ({
      ...item,
      likesCount: Array.isArray(item.likes) ? item.likes.length : 0,
      type: "story",
    }));
    
    // Randomize the results if requested
    if (randomize) {
      processedStories = shuffleArray(processedStories);
    }

    console.timeEnd("fetchAllStories");

    res.status(200).json({
      success: true,
      stories: processedStories,
      total
    });
  } catch (error) {
    console.error("Error fetching all stories:", error);
    res.status(500).json({
      success: false,
      stories: [],
      error: error.message
    });
  }
};

// Helper function to shuffle array
const shuffleArray = (array) => {
  if (!array || array.length === 0) return [];
  const newArray = [...array];
  for (let i = newArray.length - 1; i > 0; i--) {
    const j = Math.floor(Math.random() * (i + 1));
    [newArray[i], newArray[j]] = [newArray[j], newArray[i]];
  }
  return newArray;
};

// Optimize getAllPublishedStories to load all data at once
const getAllPublishedStories = async (req, res) => {
  try {
    const { language, sortBy = "reads", sortOrder = "desc" } = req.query;
    
    console.time('fetchPublishedStories');
    
    // Create the base query
    const query = { status: "published" };
    if (language) query.language = language;
    
    // Use projection to only fetch needed fields
    const stories = await Story.find(query)
      .select('title coverImage summary author categories likes reads createdAt averageRating language')
      .populate('author', 'username profileImage')
      .sort({ reads: -1, createdAt: -1 })
      .lean();
    
    console.timeEnd('fetchPublishedStories');
    
    // Return all stories without pagination
    res.status(200).json({
      success: true,
      stories: stories.map(story => ({
        ...story,
        likesCount: Array.isArray(story.likes) ? story.likes.length : 0,
        type: "story"
      })),
      total: stories.length
    });
  } catch (error) {
    console.error("Error fetching published stories:", error);
    res.status(200).json({ success: false, stories: [] });
  }
};








//? Review function
// Add a review to a story
// const addReview = async (req, res) => {
//   try {
//     const { rating, content } = req.body;
//     if (!content || !content.trim()) {
//       return res.status(400).json({ error: "Review content is required" });
//     }

//     const user = await User.findById(req.userId);
//     const story = await Story.findById(req.params.id);
//     if (!story) return res.status(404).json({ error: "Story not found" });


//      // Skip notification if reviewer is the author
//      if (story.author.toString() !== req.userId) {
//       await createNotification(story.author, {
//         type: "review",
//         message: `${user.username} reviewed your story "${story.title}"`,
//         fromUser: req.userId,
//         contentId: story._id,
//         contentModel: "Story",
//         contentAuthorId: story.author
//       });
//     }
//     // Add the review
//     story.reviews.push({
//       user: req.userId,
//       rating,
//       content: content.trim(),
//       likes: [],
//       replies: [],
//     });
//     story.reviewedUsers.push(req.userId);

//     await story.save();

//     // Add notification for review
//     await User.findByIdAndUpdate(story.author, {
//       $push: {
//         notifications: {
//           type: "review",
//           message: `${user.username} reviewed your story "${story.title}"`,
//           fromUser: req.userId,
//           contentId: story._id,
//           contentModel: "Story",
//           isRead: false,
//           createdAt: new Date(),
//         },
//       },
//     });

//     // Return populated review data
//     const populatedStory = await Story.findById(story._id)
//       .populate("reviews.user", "username")
//       .populate("reviews.replies.user", "username");

//     res.status(201).json({
//       success: true,
//       message: "Review added successfully",
//       story: populatedStory,
//     });
//   } catch (error) {
//     res.status(500).json({ error: error.message });
//   }
// };

const addReview = async (req, res) => {
  try {
    const { rating, content } = req.body;
    if (!content || !content.trim()) {
      return res.status(400).json({ error: "Review content is required" });
    }

    const user = await User.findById(req.userId);
    const story = await Story.findById(req.params.id);
    if (!story) return res.status(404).json({ error: "Story not found" });

    // Direct comparison of author ID with reviewer ID
    const isAuthor = story.author.toString() === req.userId;
    
    // Only create notification if reviewer is not the author
    if (!isAuthor) {
      await createNotification(story.author, {
        type: "review",
        message: `${user.username} reviewed your story "${story.title}"`,
        fromUser: req.userId,
        contentId: story._id,
        contentModel: "Story",
        contentAuthorId: story.author
      });
    }

    story.reviews.push({
      user: req.userId,
      rating,
      content: content.trim(),
      likes: [],
      replies: []
    });
    story.reviewedUsers.push(req.userId);

    await story.save();

    const populatedStory = await Story.findById(story._id)
      .populate("reviews.user", "username")
      .populate("reviews.replies.user", "username");

    res.status(201).json({
      success: true,
      message: "Review added successfully",
      story: populatedStory
    });
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};

// Add reply to a review
const addReplyToReview = async (req, res) => {
  try {
    const { content } = req.body;
    const story = await Story.findById(req.params.id);
    if (!story) return res.status(404).json({ error: "Story not found" });

    const review = story.reviews.id(req.params.reviewId);
    if (!review) return res.status(404).json({ error: "Review not found" });

    const replier = await User.findById(req.userId);
    review.replies.push({
      user: req.userId,
      content,
      likes: [],
    });

    await story.save();

    // Add notification with reviewAuthorId
    await createNotification(review.user, {
      type: "reply",
      message: `${replier.username} replied to your review on "${story.title}"`,
      fromUser: req.userId,
      contentId: story._id,
      contentModel: "Story",
      reviewAuthorId: review.user // Added this line
    });

    res.status(201).json({
      success: true,
      message: "Reply added successfully",
      story,
    });
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};


// Like a review
const likeReview = async (req, res) => {
  try {
    const { id, reviewId } = req.params;
    const story = await Story.findById(id);
    if (!story) return res.status(404).json({ error: "Story not found" });

    const review = story.reviews.id(reviewId);
    if (!review) return res.status(404).json({ error: "Review not found" });

    const likeIndex = review.likes.indexOf(req.userId);
    if (likeIndex === -1) {
      review.likes.push(req.userId);
    } else {
      review.likes.splice(likeIndex, 1);
    }

    await story.save();

    res.status(200).json({
      success: true,
      message: "Review like toggled successfully",
      likesCount: review.likes.length,
    });
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};

// Like a reply
const likeReply = async (req, res) => {
  try {
    const { id, reviewId, replyId } = req.params;
    const story = await Story.findById(id);
    if (!story) return res.status(404).json({ error: "Story not found" });

    const review = story.reviews.id(reviewId);
    if (!review) return res.status(404).json({ error: "Review not found" });

    const reply = review.replies.id(replyId);
    if (!reply) return res.status(404).json({ error: "Reply not found" });

    const likeIndex = reply.likes.indexOf(req.userId);
    if (likeIndex === -1) {
      reply.likes.push(req.userId);
    } else {
      reply.likes.splice(likeIndex, 1);
    }

    await story.save();

    const populatedStory = await Story.findById(story._id)
      .populate("reviews.user", "username")
      .populate("reviews.replies.user", "username")
      .populate("reviews.replies.mentionedUser", "username");

    res.status(200).json({
      success: true,
      story: populatedStory,
    });
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};

// Get all reviews for a story
const getStoryReviews = async (req, res) => {
  try {
    const story = await Story.findById(req.params.id)
      .populate("reviews.user", "username")
      .populate("reviews.replies.user", "username")
      .select("reviews");

    if (!story) return res.status(404).json({ error: "Story not found" });

    res.status(200).json({
      success: true,
      reviews: story.reviews,
      message: "Reviews fetched successfully",
    });
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};

// Get specific review with its replies
const getReviewWithReplies = async (req, res) => {
  try {
    const { id, reviewId } = req.params;
    const story = await Story.findById(id)
      .populate("reviews.user", "username")
      .populate("reviews.replies.user", "username");

    if (!story) return res.status(404).json({ error: "Story not found" });

    const review = story.reviews.id(reviewId);
    if (!review) return res.status(404).json({ error: "Review not found" });

    res.status(200).json({
      success: true,
      review,
      message: "Review and replies fetched successfully",
    });
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};

const addNestedReply = async (req, res) => {
  try {
    const { id, reviewId, replyId } = req.params;
    const { content } = req.body;

    const story = await Story.findById(id);
    if (!story) return res.status(404).json({ error: "Story not found" });

    const review = story.reviews.id(reviewId);
    if (!review) return res.status(404).json({ error: "Review not found" });

    const reply = review.replies.id(replyId);
    if (!reply) return res.status(404).json({ error: "Reply not found" });

    const newReply = {
      user: req.userId,
      content,
      replies: [],
    };

    reply.replies.push(newReply);
    await story.save();

    const populatedStory = await Story.findById(story._id)
      .populate("reviews.user", "username")
      .populate("reviews.replies.user", "username")
      .populate("reviews.replies.replies.user", "username");

    res.status(201).json({
      success: true,
      message: "Nested reply added successfully",
      story: populatedStory,
    });
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};

const updateReview = async (req, res) => {
  try {
    const { id, reviewId } = req.params;
    const { content, rating } = req.body;

    const story = await Story.findById(id);
    if (!story) return res.status(404).json({ error: "Story not found" });

    const review = story.reviews.id(reviewId);
    if (!review) return res.status(404).json({ error: "Review not found" });

    //  Check if user own the review
    if (String(review.user) !== String(req.userId)) {
      return res
        .status(403)
        .json({ error: "You are not authorized to update this review" });
    }

    review.content = content;
    if (rating) review.rating = rating;

    await story.save();
    const populatedStory = await Story.findById(story._id)
      .populate("reviews.user", "username")
      .populate("reviews.replies.user", "username");

    res.status(200).json({
      success: true,
      message: "Review updated successfully",
      story: populatedStory,
    });
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};

const deleteReview = async (req, res) => {
  try {
    const { id, reviewId } = req.params;
    const story = await Story.findById(id);
    if (!story) return res.status(404).json({ error: "Story not found" });

    const review = story.reviews.id(reviewId);
    if (!review) return res.status(404).json({ error: "Review not found" });

    if (String(review.user) !== String(req.userId)) {
      return res.status(403).json({ error: "You are not authorized to delete this review" });
    }

    // Remove review
    story.reviews.pull(reviewId);
    story.reviewedUsers.pull(req.userId);

    // Recalculate average rating
    const remainingReviews = story.reviews;
    if (remainingReviews.length > 0) {
      const totalRating = remainingReviews.reduce((sum, rev) => sum + rev.rating, 0);
      story.averageRating = (totalRating / remainingReviews.length).toFixed(1);
    } else {
      story.averageRating = 0;
    }

    await story.save();
    
    res.status(200).json({
      success: true,
      message: "Review deleted successfully",
      story: {
        ...story.toObject(),
        averageRating: story.averageRating
      }
    });
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};


const updateReply = async (req, res) => {
  try {
    const { id, reviewId, replyId } = req.params;
    const { content } = req.body;

    const story = await Story.findById(id);
    if (!story) return res.status(404).json({ error: "Story not found" });

    const review = story.reviews.id(reviewId);
    if (!review) return res.status(404).json({ error: "Review not found" });

    const reply = review.replies.id(replyId);
    if (!reply) return res.status(404).json({ error: "Reply not found" });

    if (String(reply.user) !== String(req.userId)) {
      return res
        .status(403)
        .json({ error: "You are not authorized to update this reply" });
    }

    reply.content = content;
    await story.save();

    const populatedStory = await Story.findById(story._id)
      .populate("reviews.user", "username")
      .populate("reviews.replies.user", "username");

    res.status(200).json({
      success: true,
      message: "Reply updated successfully",
      story: populatedStory,
    });
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};

const deleteReply = async (req, res) => {
  try {
    const { id, reviewId, replyId } = req.params;

    const story = await Story.findById(id);
    if (!story) return res.status(404).json({ error: "Story not found" });

    const review = story.reviews.id(reviewId);
    if (!review) return res.status(404).json({ error: "Review not found" });

    const reply = review.replies.id(replyId);
    if (!reply) return res.status(404).json({ error: "Reply not found" });

    if (String(reply.user) !== String(req.userId)) {
      return res
        .status(403)
        .json({ error: "You are not authorized to delete this reply" });
    }

    review.replies.pull(replyId);
    await story.save();
    res.status(200).json({
      success: true,
      message: "Reply deleted successfully",
    });
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};
const incrementReads = async (req, res) => {
  try {
    const story = await Story.findById(req.params.id);
    if (!story) return res.status(404).json({ error: "Story not found" });

    // Only increment reads for published stories
    if (story.status === "published") {
      const HOURS_24 = 24 * 60 * 60 * 1000;
      const now = new Date();
      const userReadHistory = story.readHistory.find(
        (record) => record.user.toString() === req.userId
      );

      if (!userReadHistory) {
        story.reads += 1;
        story.readHistory.push({
          user: req.userId,
          lastRead: now,
        });
      } else {
        const timeSinceLastRead = now - new Date(userReadHistory.lastRead);
        if (timeSinceLastRead >= HOURS_24) {
          story.reads += 1;
          userReadHistory.lastRead = now;
        }
      }

      await story.save();
    }

    res.status(200).json({
      success: true,
      reads: story.reads,
    });
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};

const getStoriesByCategory = async (req, res) => {
  try {
    const { category } = req.params;
    const stories = await Story.find({
      categories: category,
      status: "published",
    })
      .populate("author", "username profileImage")
      .sort({ createdAt: -1 });

    res.status(200).json({
      success: true,
      stories,
    });
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};

const tipAuthor = async (req, res) => {
  try {
    const { amount } = req.body;
    const storyId = req.params.id;
    
    // Validate amount is one of the allowed values
    const allowedAmounts = [1, 5, 8, 10];
    if (!allowedAmounts.includes(Number(amount))) {
      return res.status(400).json({ 
        error: "Invalid tip amount. Allowed values are 1, 5, 8, or 10 coins." 
      });
    }
    
    // Find the story and its author
    const story = await Story.findById(storyId);
    if (!story) {
      return res.status(404).json({ error: "Story not found" });
    }
    
    // Get the reader (current user) and author
    const reader = await User.findById(req.userId);
    const author = await User.findById(story.author);
    
    if (!reader || !author) {
      return res.status(404).json({ error: "User not found" });
    }
    
    // Check if reader has enough coins
    if ((reader.coins || 0) < amount) {
      return res.status(400).json({ 
        error: `Insufficient coins. You have ${reader.coins || 0} coins, but tried to tip ${amount} coins.` 
      });
    }
    
    // Prevent self-tipping
    if (reader._id.toString() === author._id.toString()) {
      return res.status(400).json({ error: "You cannot tip your own story." });
    }
    
    // Create transaction record
    const transaction = {
      amount: Number(amount),
      from: reader._id,
      fromUsername: reader.username,
      to: author._id,
      storyId: story._id,
      storyTitle: story.title,
      transactionType: "tip",
      transactionDate: new Date()
    };
    
    // Update reader's coin balance (deduct coins)
    await User.findByIdAndUpdate(
      reader._id,
      { 
        $inc: { 
          coins: -amount
          // Remove coinsSpent increment from here
        } 
      }
    );
    
    // Update author's earned coins (add coins)
    await User.findByIdAndUpdate(
      author._id,
      { 
        $inc: { earnedCoins: amount },
        $push: { coinTransactions: transaction }
      }
    );
    
    // Create notification for the author
    await createNotification(author._id, {
      type: "tip",
      message: `${reader.username} tipped you ${amount} coins for your story "${story.title}"`,
      fromUser: reader._id,
      contentId: story._id,
      contentModel: "Story",
      isRead: false,
    });
    
    // Update reader level based on coins spent
    try {
      // Call the updateReaderLevel function to update the reader's level
      const { updateReaderLevel } = require("../utils/readerLevelUtils");
      await updateReaderLevel(req.userId, amount);
    } catch (levelError) {
      console.error("Error updating reader level:", levelError);
      // Don't fail the whole transaction if just the level update fails
    }
    
    // Return success response with updated coin balance
    res.status(200).json({
      success: true,
      message: `Successfully tipped ${amount} coins to the author!`,
      newBalance: (reader.coins || 0) - amount
    });
    
  } catch (error) {
    console.error("Error in tip author:", error);
    res.status(500).json({ error: error.message || "Failed to process tip" });
  }
};


// Add this function to your existing Story_Controller.js






module.exports = {
  createStory,
  getUserStories,
  updateStory,
  getStoryDrafts,
  deleteStory,
  deleteStoryDraft,
  getStoryById,
  likeStory,
  getAllPublishedStories,
  addReview,
  addReplyToReview,
  likeReview,
  likeReply,
  getStoryReviews,
  getReviewWithReplies,
  addNestedReply,
  updateReview,
  deleteReview,
  updateReply,
  deleteReply,
  incrementReads,
  getStoriesByCategory,
  tipAuthor,
  getAllStories,

  // rateStory,
  // incrementViews,
  // addReview,
  // updateReview,
  // deleteReview,
  // addReply,
  // deleteReply,
  // likeReview,
  // markReviewHelpful, // Add this
  // reportReview,
  // addNestedReply,
  // addComment,
  // deleteComment,
  // getStoryReviews
};
