const User = require("../Model/User_Model");
const Story = require("../Model/Story_Model");
const Series = require("../Model/Series_Model");
const bcrypt = require("bcryptjs");
const jwt = require("jsonwebtoken");
const { OAuth2Client, Impersonated } = require("google-auth-library");
const crypto = require("crypto");
const { createNotification, removeNotification, createPaymentNotification } = require("../utils/notifications");
const nodemailer = require('nodemailer');
const { error } = require("console");
const {updateReaderLevel , getReaderLevelInfo } = require("../utils/readerLevelUtils")
// Add this at the top of your file
const NodeCache = require('node-cache');
const profileCache = new NodeCache({ stdTTL: 300 }); // Cache for 5 minutes
const client = new OAuth2Client(process.env.GOOGLE_CLIENT_ID);
const coinController = require("../Controller/coinController")
// Configure nodemailer
const transporter = nodemailer.createTransport({
  service: 'gmail',
  auth: {
    user: process.env.EMAIL_USER || 'novellhub19@gmail.com',
    pass: process.env.EMAIL_PASSWORD // Add this to your .env file
  }
});

// Helper function to send emails
const sendEmail = async (options) => {
  const mailOptions = {
    from: 'NovellHub <novellhub19@gmail.com>',
    to: options.to,
    subject: options.subject,
    html: options.html
  };

  return transporter.sendMail(mailOptions);
};


const signup = async (req, res) => {
  try {
    const { username, email, password } = req.body;
    const hashedPassword = await bcrypt.hash(password, 10);

    const user = new User({
      username,
      email,
      password: hashedPassword,
      coins: 0  // Explicitly set to 0 to override the default
    });
    await user.save();
    
    // Grant initial coins to the new user
    await coinController.grantInitialCoins(user._id);
    
    const token = jwt.sign({ userId: user._id }, process.env.JWT_SECRET);
    res.status(201).json({
      token,
      user: { id: user._id, username, email },
      message: "User Registered successfully",
    });
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};

// Add this new function to handle phone number updates
const updatePhoneNumber = async (req, res) => {
  try {
    const { phoneNumber } = req.body;
    
    // Validate phone number (Indian format)
    if (!/^[6-9]\d{9}$/.test(phoneNumber)) {
      return res.status(400).json({ error: "Invalid phone number format. Please enter a valid 10-digit Indian mobile number." });
    }
    
    // Update the user with phone number and mark onboarding as completed
    const user = await User.findByIdAndUpdate(
      req.userId,
      { 
        phoneNumber,
        hasCompletedOnboarding: true 
      },
      { new: true }
    ).select("-password");
    
    if (!user) {
      return res.status(404).json({ error: "User not found" });
    }
    
    res.status(200).json({
      success: true,
      message: "Phone number updated successfully",
      user
    });
  } catch (error) {
    console.error("Error updating phone number:", error);
    res.status(500).json({ error: error.message });
  }
};

// Add this function to check if user has completed onboarding
const checkOnboardingStatus = async (req, res) => {
  try {
    const user = await User.findById(req.userId).select("hasCompletedOnboarding phoneNumber");
    
    if (!user) {
      return res.status(404).json({ error: "User not found" });
    }
    
    res.status(200).json({
      success: true,
      hasCompletedOnboarding: user.hasCompletedOnboarding,
      hasPhoneNumber: !!user.phoneNumber
    });
  } catch (error) {
    console.error("Error checking onboarding status:", error);
    res.status(500).json({ error: error.message });
  }
};

// Update the signin function to include onboarding status
const signin = async (req, res) => {
  try {
    const { email, password } = req.body;
    const user = await User.findOne({ email });

    if (!user) {
      return res.status(404).json({ error: "User not found" });
    }
    const isValidPassword = await bcrypt.compare(password, user.password);
    if (!isValidPassword)
      return res.status(401).json({ error: "Invalid password" });

    const token = jwt.sign({ userId: user._id }, process.env.JWT_SECRET);
    res.status(201).json({
      token,
      user: { 
        id: user._id, 
        username: user.username, 
        email,
        hasCompletedOnboarding: user.hasCompletedOnboarding,
        hasPhoneNumber: !!user.phoneNumber
      },
      message: "User Logged in successfully",
    });
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};

// Update the googleAuth function to include onboarding status
const googleAuth = async (req, res) => {
  try {
    const { credential } = req.body;
    
    if (!credential) {
      return res.status(400).json({ error: "No credential provided" });
    }

    const ticket = await client.verifyIdToken({
      idToken: credential,
      audience: process.env.GOOGLE_CLIENT_ID
    });

    const payload = ticket.getPayload();
    const { email, name, picture } = payload;

    let user = await User.findOne({ email });

    if (!user) {
      user = new User({
        username: name,
        email,
        profileImage: picture,
        password: crypto.randomBytes(16).toString("hex"),
        coins: 0  // Explicitly set to 0 to override the default
      });
      await user.save();
      
      // Grant initial coins after user creation
      await coinController.grantInitialCoins(user._id);
    }

    const token = jwt.sign({ userId: user._id }, process.env.JWT_SECRET);

    res.status(200).json({
      token,
      user: { 
        id: user._id, 
        username: user.username, 
        email: user.email,
        hasCompletedOnboarding: user.hasCompletedOnboarding,
        hasPhoneNumber: !!user.phoneNumber
      },
      message: "Google authentication successful",
    });
  } catch (error) {
    console.error("Google Auth Error:", error);
    res.status(500).json({ error: "Authentication failed" });
  }
};




// const updateProfile = async (req, res) => {
//   try {
//     const { username, bio, profileImage, coverImage } = req.body;
//     const updateUser = await User.findByIdAndUpdate(
//       req.userId,
//       { username, bio, profileImage, coverImage },
//       { new: true }
//     ).select("-password");

//     res.status(201).json({
//       success: true,
//       message: "Profile updated successfully",
//       user: updateUser,
//     });
//   } catch (error) {
//     res.status(500).json({ error: error.message });
//   }
// };
const updateProfileImage = async (req, res) => {
  try {
    if (!req.file) {
      return res.status(400).json({ error: "No file uploaded" });
    }

    const imageUrl = `/uploads/${req.file.filename}`;

    const user = await User.findByIdAndUpdate(
      req.userId,
      { profileImage: imageUrl },
      { new: true }
    ).select("-password");

    res.status(200).json({
      success: true,
      message: "Profile image updated successfully",
      user,
    });
  } catch (error) {
    res.status(500).json({ error: "Failed to update profile image" });
  }
};

const updateCoverImage = async (req, res) => {
  try {
    if (!req.file) {
      return res.status(400).json({ error: "No file uploaded" });
    }

    const imageUrl = `/uploads/${req.file.filename}`;

    const user = await User.findByIdAndUpdate(
      req.userId,
      { coverImage: imageUrl },
      { new: true }
    ).select("-password");

    res.status(200).json({
      success: true,
      message: "Cover image updated successfully",
      user,
    });
  } catch (error) {
    res.status(500).json({ error: "Failed to update cover image" });
  }
};

const updateProfileDetails = async (req, res) => {
  try {
    const { username, bio } = req.body;
    const user = await User.findByIdAndUpdate(
      req.userId,
      { username, bio },
      { new: true }
    ).select("-password");

    res.status(200).json({
      success: true,
      user,
      message: "Profile details updated successfully",
    });
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};


const getProfile = async (req, res) => {
  try {
    const cacheKey = `profile_${req.userId}`;
    
    // Check if we have cached data
    const cachedProfile = profileCache.get(cacheKey);
    if (cachedProfile) {
      return res.status(200).json({
        success: true,
        user: cachedProfile,
        message: "Profile fetched successfully (cached)",
      });
    }
    
    // If no cached data, proceed with database query
    const user = await User.findById(req.userId)
      .select("-password")
      .populate("followers", "username profileImage")
      .populate("following", "username profileImage")
      .lean();

    if (!user) {
      return res.status(404).json({ error: "User not found" });
    }

    // Use Promise.all to run queries in parallel
    const [stories, series] = await Promise.all([
      Story.find({ author: req.userId }).lean(),
      Series.find({ author: req.userId }).lean()
    ]);

    // Calculate stats more efficiently
    const storyReads = stories.reduce(
      (total, story) => total + (story.reads || 0),
      0
    );
    
    const storyLikes = stories.reduce(
      (total, story) => total + (story.likes?.length || 0),
      0
    );
    
    // Calculate series stats more efficiently
    let seriesReads = 0;
    let seriesLikes = 0;
    
    series.forEach(seriesItem => {
      // Add series-level likes
      seriesLikes += (seriesItem.likes?.length || 0);
      
      // Calculate reads based on available metrics
      let seriesReadCount = 0;
      
      if (seriesItem.episodes && seriesItem.episodes.length > 0) {
        // Calculate episode reads and likes in one pass
        seriesItem.episodes.forEach(episode => {
          seriesLikes += (episode.likes?.length || 0);
          seriesReadCount += (episode.totalReaders || episode.reads || 0);
        });
      } else {
        seriesReadCount = seriesItem.totalReads || seriesItem.uniqueReads || seriesItem.reads || 0;
      }
      
      seriesReads += seriesReadCount;
    });
    
    const totalReads = storyReads + seriesReads;
    const totalLikes = storyLikes + seriesLikes;

    const userData = {
      ...user,
      stories,
      series,
      stats: {
        totalReads,
        totalLikes,
        storyReads,
        seriesReads,
        storyLikes,
        seriesLikes
      },
    };
    
    // Store in cache
    profileCache.set(cacheKey, userData);

    res.status(200).json({
      success: true,
      user: userData,
      message: "Profile fetched successfully",
    });
  } catch (error) {
    console.error("Error fetching profile:", error);
    res.status(500).json({ error: error.message });
  }
};

// Add these endpoints to split up the data fetching



// Get user stats
const getUserStats = async (req, res) => {
  try {
    const userId = req.params.userId || req.userId;
    const cacheKey = `stats_${userId}`;
    
    // Check cache first
    const cachedStats = profileCache.get(cacheKey);
    if (cachedStats) {
      return res.status(200).json({
        success: true,
        stats: cachedStats,
        message: "Stats fetched successfully (cached)",
      });
    }
    
    // Use Promise.all to run queries in parallel
    const [stories, series] = await Promise.all([
      Story.find({ author: userId }).select('reads likes').lean(),
      Series.find({ author: userId }).select('likes episodes').lean()
    ]);

    // Calculate stats efficiently
    const storyReads = stories.reduce(
      (total, story) => total + (story.reads || 0),
      0
    );
    
    const storyLikes = stories.reduce(
      (total, story) => total + (story.likes?.length || 0),
      0
    );
    
    let seriesReads = 0;
    let seriesLikes = 0;
    
    series.forEach(seriesItem => {
      seriesLikes += (seriesItem.likes?.length || 0);
      
      if (seriesItem.episodes && seriesItem.episodes.length > 0) {
        seriesItem.episodes.forEach(episode => {
          seriesReads += (episode.totalReaders || episode.reads || 0);
        });
      }
    });
    
    const stats = {
      totalReads: storyReads + seriesReads,
      totalLikes: storyLikes + seriesLikes,
      storyReads,
      seriesReads,
      storyLikes,
      seriesLikes
    };
    
    // Cache the result
    profileCache.set(cacheKey, stats);
    
    res.status(200).json({
      success: true,
      stats,
      message: "Stats fetched successfully",
    });
  } catch (error) {
    console.error("Error fetching user stats:", error);
    res.status(500).json({ error: error.message });
  }
};



// Add this new endpoint for basic profile data
const getBasicProfile = async (req, res) => {
  try {
    const user = await User.findById(req.userId)
      .select("username bio profileImage coverImage followers following accountDetails")
      .lean();

    if (!user) {
      return res.status(404).json({ error: "User not found" });
    }

    res.status(200).json({
      success: true,
      user,
      message: "Basic profile fetched successfully",
    });
  } catch (error) {
    console.error("Error fetching basic profile:", error);
    res.status(500).json({ error: error.message });
  }
};

const followUser = async (req, res) => {
  try {
    console.log("Follow user request received for:", req.params.userId);
    
    // Find users using findById to get the full documents
    const userToFollow = await User.findById(req.params.userId);
    const currentUser = await User.findById(req.userId);

    if (!userToFollow || !currentUser) {
      console.log("User not found:", !userToFollow ? "target user" : "current user");
      return res.status(404).json({ error: "User not found" });
    }

    if (userToFollow._id.toString() === currentUser._id.toString()) {
      console.log("User attempted to follow themselves");
      return res.status(400).json({ error: "Cannot follow yourself" });
    }

    // Check if already following
    const isFollowing = userToFollow.followers.some(
      id => id.toString() === req.userId
    );
    
    console.log("Current follow status:", isFollowing ? "following" : "not following");

    // Use direct MongoDB operations to update the follow status
    // This bypasses Mongoose validation issues
    if (isFollowing) {
      // Unfollow - pull the IDs from the arrays
      await User.updateOne(
        { _id: userToFollow._id },
        { $pull: { followers: req.userId } }
      );
      
      await User.updateOne(
        { _id: req.userId },
        { $pull: { following: userToFollow._id } }
      );
      
      console.log("Unfollow operation completed");
    } else {
      // Follow - add the IDs to the arrays
      await User.updateOne(
        { _id: userToFollow._id },
        { $addToSet: { followers: req.userId } }
      );
      
      await User.updateOne(
        { _id: req.userId },
        { $addToSet: { following: userToFollow._id } }
      );
      
      console.log("Follow operation completed");
      
      // Try to add notification, but don't let it fail the whole operation
      try {
        // Use updateOne to add notification directly
        await User.updateOne(
          { _id: userToFollow._id },
          { 
            $push: { 
              notifications: {
                type: "follow",
                message: `${currentUser.username} started following you`,
                fromUser: currentUser._id,
                contentId: currentUser._id,
                contentModel: "User",
                createdAt: new Date(),
                isRead: false
              }
            }
          }
        );
        console.log("Notification added successfully");
      } catch (notifError) {
        // Log the error but continue
        console.error("Failed to add notification, but follow succeeded:", notifError);
      }
    }

    // Verify the operation worked by checking the updated status
    const updatedUserToFollow = await User.findById(req.params.userId);
    const newIsFollowing = updatedUserToFollow.followers.some(
      id => id.toString() === req.userId
    );
    
    console.log("New follow status:", newIsFollowing ? "following" : "not following");
    
    res.status(200).json({
      success: true,
      isFollowing: newIsFollowing,
      message: isFollowing ? "Unfollowed successfully" : "Following successfully",
    });
  } catch (error) {
    console.error("Follow user error:", error);
    res.status(500).json({ success: false, error: error.message });
  }
};


const getStats = async (req, res) => {
  try {
    // Get unique authors from both stories and series
    const storyWriters = await Story.distinct("author", {
      status: "published",
    });
    
    const seriesWriters = await Series.distinct("author", {
      status: { $ne: "draft" }
    });

    // Combine unique writers from both stories and series
    const allWriters = [...new Set([...storyWriters, ...seriesWriters])];

    // Get unique readers from stories
    const storyReaders = await Story.distinct("readHistory.user");
    
    // Get unique readers from series episodes
    const seriesReaders = await Series.aggregate([
      { $unwind: "$episodes" },
      { $unwind: "$episodes.readers" },
      { $group: { _id: "$episodes.readers" } }
    ]);

    // Combine unique readers from both stories and series
    const totalUniqueReaders = [...new Set([
      ...storyReaders, 
      ...seriesReaders.map(r => r._id.toString())
    ])];

    // Get total published counts
    const totalPublishedStories = await Story.countDocuments({ status: "published" });
    const totalPublishedSeries = await Series.countDocuments({ status: { $ne: "draft" } });

    res.status(200).json({
      success: true,
      stats: {
        activeWriters: allWriters.length,
        readersCount: totalUniqueReaders.length,
        publishedStories: totalPublishedStories,
        totalPublished: totalPublishedStories + totalPublishedSeries
      },
    });
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};











const getNotifications = async (req, res) => {
  try {
    const user = await User.findById(req.userId).populate({
      path: "notifications.fromUser",
      select: "username profileImage",
    });
    if (!user) {
      return res.status(404).json({ error: "User not found" });
    }

    res.status(200).json({
      success: true,
      notifications: user.notifications || [],
    });
  } catch (error) {
    console.error("Notification fetch error:", error);
    res.status(500).json({ error: error.message });
  }
};

const markNotificationsAsRead = async (req, res) => {
  try {
    const user = await User.findByIdAndUpdate(
      req.userId,
      { $set: { notifications: [] } },
      { new: true }
    );

    res.status(200).json({
      success: true,
      message: "All notifications cleared",
      notifications: [],
    });
  } catch (error) {
    console.error("Clear notifications error:", error);
    res.status(500).json({ error: error.message });
  }
};

// For likes on stories/series
const createLikeNotification = async (req, res) => {
  const { authorId, contentId, contentType, title } = req.body;
  try {
    await createNotification(authorId, {
      type: "like",
      message: `${
        req.user.username
      } liked your ${contentType.toLowerCase()} "${title}"`,
      fromUser: req.userId,
      contentId,
      contentModel: contentType,
    });
    res.status(200).json({ success: true });
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};

// For reviews
const createReviewNotification = async (req, res) => {
  const { authorId, contentId, contentType, title } = req.body;
  try {
    // Skip if reviewer is content author
    if (authorId === req.userId) {
      return res.status(200).json({ success: true });
    }

    await createNotification(authorId, {
      type: "review",
      message: `${req.user.username} reviewed your ${contentType.toLowerCase()} "${title}"`,
      fromUser: req.userId,
      contentId,
      contentModel: contentType,
      contentAuthorId: authorId
    });
    res.status(200).json({ success: true });
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};


// For review replies
const createReplyNotification = async (req, res) => {
  const { recipientId, contentId, contentType, title } = req.body;
  try {
    await createNotification(recipientId, {
      type: "reply",
      message: `${req.user.username} replied to your review on "${title}"`,
      fromUser: req.userId,
      contentId,
      contentModel: contentType,
    });
    res.status(200).json({ success: true });
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};

// For new content notifications to followers
const notifyFollowers = async (authorId, contentId, contentType, title) => {
  const author = await User.findById(authorId);
  const followers = author.followers;

  const notifications = followers.map((followerId) => ({
    userId: followerId,
    notification: {
      type: "new_content",
      message: `${
        author.username
      } published a new ${contentType.toLowerCase()} "${title}"`,
      fromUser: authorId,
      contentId,
      contentModel: contentType,
      isRead: false,
    },
  }));

  await Promise.all(
    notifications.map(({ userId, notification }) =>
      createNotification(userId, notification)
    )
  );
};
const getFollowers = async (req, res) => {
  try {
    const user = await User.findById(req.params.userId).populate({
      path: "followers",
      select: "username profileImage"
    });

    if (!user) {
      return res.status(404).json({ error: "User not found" });
    }

    res.status(200).json({
      success: true,
      followers: user.followers,
    });
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};

const getFollowing = async (req, res) => {
  try {
    const user = await User.findById(req.params.userId).populate({
      path: "following",
      select: "username profileImage"
    });

    if (!user) {
      return res.status(404).json({ error: "User not found" });
    }

    res.status(200).json({
      success: true,
      following: user.following,
    });
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};


const checkFollowStatus = async (req, res) => {
  try {
    console.log("Checking follow status for user:", req.params.userId);
    
    const currentUser = await User.findById(req.userId);
    if (!currentUser) {
      return res.status(404).json({ error: "Current user not found" });
    }
    
    // Check if the current user is following the target user
    const isFollowing = currentUser.following.some(
      id => id.toString() === req.params.userId
    );
    
    console.log("Follow status check result:", isFollowing ? "following" : "not following");
    
    res.status(200).json({
      success: true,
      isFollowing,
    });
  } catch (error) {
    console.error("Error checking follow status:", error);
    res.status(500).json({ error: error.message });
  }
};

// Get user stories
const getUserStories = async (req, res) => {
  try {
    const userId = req.params.userId || req.userId;
    const cacheKey = `stories_${userId}`;
    
    // Check cache first
    const cachedStories = profileCache.get(cacheKey);
    if (cachedStories) {
      return res.status(200).json({
        success: true,
        stories: cachedStories,
        message: "Stories fetched successfully (cached)",
      });
    }
    
    const stories = await Story.find({ author: userId })
      .select('title coverImage reads likes status')
      .lean();
    
    // Cache the result
    profileCache.set(cacheKey, stories);
    
    res.status(200).json({
      success: true,
      stories,
      message: "Stories fetched successfully",
    });
  } catch (error) {
    console.error("Error fetching user stories:", error);
    res.status(500).json({ error: error.message });
  }
};

// Get user series
const getUserSeries = async (req, res) => {
  try {
    const userId = req.params.userId || req.userId;
    const cacheKey = `series_${userId}`;
    
    // Check cache first
    const cachedSeries = profileCache.get(cacheKey);
    if (cachedSeries) {
      return res.status(200).json({
        success: true,
        series: cachedSeries,
        message: "Series fetched successfully (cached)",
      });
    }
    
    const series = await Series.find({ author: userId })
      .select('title coverImage episodes likes')
      .lean();
    
    // Cache the result
    profileCache.set(cacheKey, series);
    
    res.status(200).json({
      success: true,
      series,
      message: "Series fetched successfully",
    });
  } catch (error) {
    console.error("Error fetching user series:", error);
    res.status(500).json({ error: error.message });
  }
};

const getUserProfile = async (req, res) => {
  try {
    const user = await User.findById(req.params.userId)
      .select('-password')
      .populate('followers', 'username profileImage')
      .populate('following', 'username profileImage');

    if (!user) {
      return res.status(404).json({ error: 'User not found' });
    }

    const stories = await Story.find({ 
      author: req.params.userId,
      status: 'published'
    });
    
    const series = await Series.find({ 
      author: req.params.userId,
      status: { $ne: 'draft' }
    });

    // Calculate total reads more accurately
    // 1. For individual stories
    const storyReads = stories.reduce(
      (total, story) => total + (story.reads || 0),
      0
    );
    
    // 2. For series - log detailed information
    let seriesReads = 0;
    for (const seriesItem of series) {
     
      let episodeReadsTotal = 0;
      let episodeTotalReaders = 0;
      if (seriesItem.episodes && seriesItem.episodes.length > 0) {
        
        seriesItem.episodes.forEach((episode, idx) => {
        
          episodeReadsTotal += (episode.reads || 0);
          episodeTotalReaders += (episode.totalReaders || 0);
        });
      }

      // For now, let's use totalReaders as our primary metric if available
      let seriesReadCount = 0;
      
      if (episodeTotalReaders > 0) {
        seriesReadCount = episodeTotalReaders;
      } else if (episodeReadsTotal > 0) {
        seriesReadCount = episodeReadsTotal;
      } else if (seriesItem.totalReads > 0) {
        seriesReadCount = seriesItem.totalReads;
      } else if (seriesItem.uniqueReads > 0) {
        seriesReadCount = seriesItem.uniqueReads;
      } else {
        seriesReadCount = seriesItem.reads || 0;
      }
      
      seriesReads += seriesReadCount;
    }
    
    const totalReads = storyReads + seriesReads;
    
    // Calculate total likes
    const storyLikes = stories.reduce(
      (total, story) => total + (story.likes?.length || 0),
      0
    );
    
    let seriesLikes = 0;
    for (const seriesItem of series) {
      // Add the series-level likes
      seriesLikes += (seriesItem.likes?.length || 0);
      
      // Add likes from each episode
      if (seriesItem.episodes && seriesItem.episodes.length > 0) {
        seriesItem.episodes.forEach(episode => {
          seriesLikes += (episode.likes?.length || 0);
        });
      }
    }
    
    const totalLikes = storyLikes + seriesLikes;

    const userData = {
      ...user.toObject(),
      stories,
      series,
      stats: {
        totalReads,
        totalLikes,
        storyReads,
        seriesReads,
        storyLikes,
        seriesLikes
      },
    };

    res.status(200).json({
      success: true,
      user: userData
    });
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};




const updateAccountDetails = async (req, res) => {
  try {
    console.log("Updating account details:", req.body);
    
    const { bankName, accountNumber, ifscCode, accountHolderName, panNumber, mobileNumber, isEdit } = req.body;
    
    // Additional validation
    if (!bankName || !accountNumber || !ifscCode || !accountHolderName || !panNumber || !mobileNumber) {
      return res.status(400).json({ error: "All fields are required" });
    }

    // Validate account number format (9-18 digits)
    if (!/^\d{9,18}$/.test(accountNumber)) {
      return res.status(400).json({ error: "Invalid account number format" });
    }

    // Validate IFSC code format
    if (!/^[A-Z]{4}0[A-Z0-9]{6}$/.test(ifscCode)) {
      return res.status(400).json({ error: "Invalid IFSC code format" });
    }

    // Validate account holder name (letters and spaces only)
    if (!/^[a-zA-Z\s]{2,50}$/.test(accountHolderName)) {
      return res.status(400).json({ error: "Invalid account holder name" });
    }

    // Validate PAN number format
    if (!/^[A-Z]{5}[0-9]{4}[A-Z]{1}$/.test(panNumber)) {
      return res.status(400).json({ error: "Invalid PAN number format" });
    }

    // Validate mobile number format (Indian format)
    if (!/^[6-9]\d{9}$/.test(mobileNumber)) {
      return res.status(400).json({ error: "Invalid mobile number format" });
    }

    // First check if the user has an accountDetails field
    const user = await User.findById(req.userId);
    if (!user) {
      return res.status(404).json({ error: "User not found" });
    }
    
    console.log("Current user account details:", user.accountDetails);
    
    // If accountDetails doesn't exist, initialize it first
    if (!user.accountDetails) {
      console.log("Initializing accountDetails field");
      await User.updateOne(
        { _id: req.userId },
        { $set: { accountDetails: {} } }
      );
    }

    // Now update the account details
    console.log("Updating account details with new values");
    const result = await User.updateOne(
      { _id: req.userId },
      { 
        $set: {
          "accountDetails.bankName": bankName,
          "accountDetails.accountNumber": accountNumber,
          "accountDetails.ifscCode": ifscCode,
          "accountDetails.accountHolderName": accountHolderName,
          "accountDetails.panNumber": panNumber,
          "accountDetails.mobileNumber": mobileNumber
        }
      }
    );
    
    console.log("Update result:", result);
    
    // Get the updated user
    const updatedUser = await User.findById(req.userId).select("-password");
    
    console.log("Updated user has account details:", !!updatedUser.accountDetails);
    console.log("Updated account details:", updatedUser.accountDetails);

    res.status(200).json({
      success: true,
      message: isEdit ? "Account details updated successfully" : "Account details added successfully",
      user: updatedUser
    });
  } catch (error) {
    console.error("Error updating account details:", error);
    res.status(500).json({ error: error.message || "Failed to update account details" });
  }
};







const deleteNotification = async (req, res) => {
  try {
    const { id } = req.params;
    await removeNotification(req.userId, id);
    
    res.status(200).json({
      success: true,
      message: "Notification deleted successfully"
    });
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};


// Coin systum 
const redeemCoins = async (req, res) => {
  try {
    console.log("Redemption request received:", req.body);
    const { amount } = req.body;
    
    // Validate amount
    if (!amount || amount < 200) {
      return res.status(400).json({ 
        error: "Minimum 200 coins required for redemption" 
      });
    }
    
    // Convert amount to number to ensure proper calculations
    const amountNum = Number(amount);
    
    // Find user with more detailed logging
    const user = await User.findById(req.userId);
    if (!user) {
      return res.status(404).json({ error: "User not found" });
    }
    
    // More detailed logging
    console.log("User ID for redemption:", req.userId);
    console.log("User document keys:", Object.keys(user._doc));
    console.log("User earnedCoins:", user.earnedCoins);
    console.log("User accountDetails:", user.accountDetails);
    console.log("Type of accountDetails:", typeof user.accountDetails);
    console.log("Is accountDetails null?", user.accountDetails === null);
    console.log("Is accountDetails undefined?", user.accountDetails === undefined);
    
    // Check if user has enough earned coins
    if ((user.earnedCoins || 0) < amountNum) {
      return res.status(400).json({ 
        error: `Insufficient earned coins for redemption. You have ${user.earnedCoins || 0} coins, but requested to redeem ${amountNum} coins.` 
      });
    }

    // Enhanced check for account details
    if (!user.accountDetails || typeof user.accountDetails !== 'object') {
      // Try to fix the user document by initializing accountDetails if it doesn't exist
      await User.updateOne(
        { _id: req.userId },
        { $set: { accountDetails: {} } }
      );
      
      return res.status(400).json({
        error: "Please add your bank account details before redeeming coins",
        details: "Account details field is missing or invalid"
      });
    }
    
    // Check if all required account details are present with more detailed error
    const requiredFields = ['bankName', 'accountNumber', 'ifscCode', 'accountHolderName', 'panNumber', 'mobileNumber'];
    const missingFields = requiredFields.filter(field => !user.accountDetails[field]);

    if (missingFields.length > 0) {
      return res.status(400).json({
        error: `Incomplete account details. Missing: ${missingFields.join(', ')}`,
        details: {
          requiredFields,
          missingFields,
          accountDetailsKeys: Object.keys(user.accountDetails)
        }
      });
    }
    
    // Calculate redemption value (4 coins = ₹1.00)
    const fullRedemptionValue = amountNum / 4;
    
    // Apply 20% platform fee - writer gets 80%
    const writerShare = fullRedemptionValue * 0.8;
    const platformFee = fullRedemptionValue * 0.2;
    
    // Create redemption record
    const redemption = {
      amount: amountNum,
      value: writerShare,
      fullValue: fullRedemptionValue,
      platformFee: platformFee,
      status: "pending",
      requestedAt: new Date()
    };
    
    // Use direct MongoDB update to ensure atomic operation
    const result = await User.updateOne(
      { _id: req.userId },
      { 
        $inc: { earnedCoins: -amountNum },
        $push: { coinRedemptions: redemption }
      }
    );
    
    console.log("Update result:", result);
    
    // Get updated user data
    const updatedUser = await User.findById(req.userId);
    
    // Get the newly added redemption
    const newRedemption = updatedUser.coinRedemptions[updatedUser.coinRedemptions.length - 1];
    
    res.status(200).json({
      success: true,
      message: `Redemption request submitted successfully. You will receive ₹${writerShare.toFixed(2)} (after 20% platform fee) in your bank account within 3-5 business days.`,
      redemption: newRedemption,
      currentEarnedCoins: updatedUser.earnedCoins
    });
  } catch (error) {
    console.error('Error in coin redemption:', error);
    res.status(500).json({ error: error.message || "Failed to redeem coins" });
  }
};










// Update the getRedemptionHistory function
const getRedemptionHistory = async (req, res) => {
  try {
    
    // Get the user
    const user = await User.findById(req.userId);
    if (!user) {
      return res.status(404).json({ error: "User not found" });
    }
    

    
    // Ensure coinRedemptions is an array
    if (!Array.isArray(user.coinRedemptions)) {
      // Update the user document to initialize coinRedemptions as an empty array
      await User.updateOne(
        { _id: req.userId },
        { $set: { coinRedemptions: [] } }
      );
      
      // Get the updated user
      const updatedUser = await User.findById(req.userId);
      
      res.status(200).json({
        success: true,
        redemptions: [],
        message: "Redemption history initialized successfully"
      });
    } else {
      res.status(200).json({
        success: true,
        redemptions: user.coinRedemptions || [],
        message: "Redemption history retrieved successfully"
      });
    }
  } catch (error) {
    console.error("Error fetching redemption history:", error);
    res.status(500).json({ error: error.message });
  }
};






// Update the getCoinBalance function
const getCoinBalance = async (req, res) => {
  try {
    // Use findById with lean() to get the raw data without Mongoose document methods
    const user = await User.findById(req.userId).lean();
    
    if (!user) {
      return res.status(404).json({ error: "User not found" });
    }
    

    // Return both types of coin balances
    res.status(200).json({
      success: true,
      coins: user.coins || 0,           // Default/reader coins
      earnedCoins: user.earnedCoins || 0, // Writer earnings
      totalCoins: (user.coins || 0) + (user.earnedCoins || 0), // Total available
      message: "Coin balance retrieved successfully"
    });
  } catch (error) {
    console.error("Error getting coin balance:", error);
    res.status(500).json({ error: error.message });
  }
};




// Add these new admin functions to handle redemption requests
const getRedemptionRequests = async (req, res) => {
  try {
    // Remove this admin check since we're using adminAuth middleware
    // const admin = await User.findById(req.userId);
    // if (!admin || admin.role !== 'admin') {
    //   return res.status(403).json({ error: "Unauthorized access" });  
    // }

    // Get all redemption requests with user details
    const users = await User.find({ 'coinRedemptions.0': { $exists: true } })
      .select('username email coinRedemptions accountDetails');

    // Format the data for frontend  
    const allRedemptions = [];
    
    users.forEach(user => {
      user.coinRedemptions.forEach(redemption => {
        allRedemptions.push({
          _id: redemption._id,
          amount: redemption.amount,
          value: redemption.value,
          status: redemption.status,
          requestedAt: redemption.requestedAt,
          processedAt: redemption.processedAt,
          user: {
            _id: user._id,
            username: user.username,
            email: user.email
          },  
          accountDetails: user.accountDetails
        });  
      });  
    });  

    // Sort by date (newest first)
    allRedemptions.sort((a, b) => new Date(b.requestedAt) - new Date(a.requestedAt));

    res.status(200).json({
      success: true,
      redemptionRequests: allRedemptions
    });  
  } catch (error) {
    console.error('Error fetching redemption requests:', error);
    res.status(500).json({ error: error.message });
  }  
};  

// Update the processRedemption function
const processRedemption = async (req, res) => {
  try {
    const { redemptionId, action } = req.body;

    if (!['approve', 'reject'].includes(action)) {
      return res.status(400).json({ error: "Invalid action" });
    }

    // Find the user with the redemption
    // First, find which user has this redemption ID
    const users = await User.find({ 'coinRedemptions._id': redemptionId });
    
    if (!users || users.length === 0) {
      return res.status(404).json({ error: "Redemption request not found" });
    }
    
    const user = users[0]; // Get the first user that has this redemption

    // Find the specific redemption
    const redemptionIndex = user.coinRedemptions.findIndex(
      r => r._id.toString() === redemptionId
    );

    if (redemptionIndex === -1) {
      return res.status(404).json({ error: "Redemption request not found" });
    }

    // Update the redemption status
    user.coinRedemptions[redemptionIndex].status = action === 'approve' ? 'approved' : 'rejected';
    user.coinRedemptions[redemptionIndex].processedAt = new Date();

    // Get the payment amount for notification
    const paymentAmount = (user.coinRedemptions[redemptionIndex].value || 
                          (user.coinRedemptions[redemptionIndex].amount / 4 * 0.8)).toFixed(2);

    await user.save();

    // Create a notification for the user
    await createPaymentNotification(
      user._id, 
      action === 'approve' ? 'approved' : 'rejected',
      paymentAmount
    );

    // Send email notification to the user
    const emailContent = `
      <h2>Your Coin Redemption Request ${action === 'approve' ? 'Approved' : 'Rejected'}</h2>
      <p>Dear ${user.username},</p>
      <p>Your request to redeem ${user.coinRedemptions[redemptionIndex].amount} coins has been ${action === 'approve' ? 'approved' : 'rejected'}.</p>
      ${action === 'approve' ? 
        `<p>You will receive ₹${paymentAmount} in your bank account within 3-5 business days.</p>` : 
        `<p>If you have any questions about why your request was rejected, please contact our support team.</p>`
      }
      <p>Thank you for being a valued writer on NovellHub!</p>
    `;

    try {
      await sendEmail({
        to: user.email,
        subject: `Redemption Request ${action === 'approve' ? 'Approved' : 'Rejected'} - NovellHub`,
        html: emailContent
      });
    } catch (emailError) {
      console.error('Error sending redemption status email:', emailError);
      // Continue with the process even if email fails
    }

    res.status(200).json({
      success: true,
      message: `Redemption request ${action === 'approve' ? 'approved' : 'rejected'} successfully`
    });
  } catch (error) {
    console.error('Error processing redemption:', error);
    res.status(500).json({ error: error.message });
  }
};





const grantInitialCoinsToAllUsers = async (req, res) => {
  try {
    // Find all users who don't have the 'initialCoinGranted' flag set
    const result = await User.updateMany(
      { initialCoinGranted: { $ne: true } },
      { 
        $set: { 
          initialCoinGranted: true,
          coins: 25  // Changed from 20 to 25 coins
        }
      }
    );

    res.status(200).json({
      success: true,
      message: `Initial coins granted to ${result.modifiedCount} users`,
      usersUpdated: result.modifiedCount
    });
  } catch (error) {
    console.error("Error granting initial coins:", error);
    res.status(500).json({ error: error.message });
  }
};

// Add this middleware function to update last active timestamp
const updateLastActive = async (req, res, next) => {
  try {
    if (req.userId) {
      await User.findByIdAndUpdate(req.userId, { lastActive: new Date() });
    }
    next();
  } catch (error) {
    next();
  }
};




const adminImpersonateUser = async (req, res) => {
  try {
    console.log("Admin impersonation request received");
    
    // Check if the request is authenticated as an admin
    if (!req.isAdmin) {
      return res.status(403).json({ error: "Unauthorized access" });
    }
    
    const { writerId } = req.params;
    console.log("Writer ID to impersonate:", writerId);

    // Find the writer to impersonate
    const writer = await User.findById(writerId);
    if (!writer) {
      return res.status(404).json({ error: "Writer not found" });
    }
     
    // Generate a special impersonation token with limited expiry
    const impersonationToken = jwt.sign(
      { 
        userId: writer._id, 
        impersonatedBy: req.adminId || req.userId,
        isImpersonation: true
      }, 
      process.env.JWT_SECRET,
      { expiresIn: '2h' }
    );
    
    // Log this action for security audit
    const adminIdentifier = req.admin ? req.admin.username : (req.user ? req.user.username : "Unknown admin");
    const adminId = req.adminId || req.userId;
    
    console.log(`Admin ${adminIdentifier} (${adminId}) impersonated writer ${writer.username} (${writerId}) at ${new Date().toISOString()}`);
    
    res.status(200).json({
      success: true,
      impersonationToken,
      writer: {
        id: writer._id,
        username: writer.username,
        email: writer.email
      }
    });
  } catch (error) {
    console.error("Error in admin impersonation:", error);
    res.status(500).json({ error: error.message });
  }
};


// Add these functions to the User_Controller.js file





// Function to get top readers for the Wall of Fame
const getReaderWallOfFame = async (req, res) => {
  try {
    // Get top platinum readers (limited to 20)
    const platinumReaders = await User.find({ 
      readerLevel: 'Platinum' 
    })
    .sort({ coinsSpent: -1 })
    .limit(20)
    .select('username profileImage readerLevel coinsSpent');
    
    // Get top gold readers (limited to 10)
    const goldReaders = await User.find({ 
      readerLevel: 'Gold' 
    })
    .sort({ coinsSpent: -1 })
    .limit(10)
    .select('username profileImage readerLevel coinsSpent');
    
    res.status(200).json({
      success: true,
      wallOfFame: {
        platinum: platinumReaders,
        gold: goldReaders
      }
    });
  } catch (error) {
    console.error("Error fetching reader wall of fame:", error);
    res.status(500).json({ error: error.message });
  }
};

// Function to update coins spent when a user spends coins
const updateCoinsSpent = async (req, res) => {
  try {
    const { amount } = req.body;
    
    if (!amount || isNaN(amount) || amount <= 0) {
      return res.status(400).json({ error: "Invalid amount" });
    }
    
    // Update reader level and coins spent
    const result = await updateReaderLevel(req.userId, amount);
    
    if (!result) {
      return res.status(404).json({ error: "User not found" });
    }
    
    res.status(200).json({
      success: true,
      message: `Coins spent updated successfully`,
      readerLevel: result.level,
      coinsSpent: result.coinsSpent,
      badges: result.badges,
      levelUp: result.levelUp,
      bonusCoins: result.bonusCoins
    });
  } catch (error) {
    console.error("Error updating coins spent:", error);
    res.status(500).json({ error: error.message });
  }
};

const getUserReaderLevelData = async (req, res) => {
  try {
    const readerLevel = await getReaderLevelInfo(req.userId);
    
    if (!readerLevel) {
      return res.status(404).json({ error: "User not found" });
    }
    
    res.status(200).json({
      success: true,
      readerLevel
    });
  } catch (error) {
    console.error("Error getting reader level:", error);
    res.status(500).json({ error: error.message });
  }
};

const getUserReaderLevel  = async (req,res)=>{
  try {
    const {userId} = req.params;
    // validate that userId is provided 
    if(!userId){
      return res.status(400).json({error:"userId is required"})
    }
    // Get reader level info for the specified user 
    const readerLevel = await getReaderLevelInfo(userId);
    if(!readerLevel){
      return res.status(404).json({error:"User not found"})
    }
    res.status(200).json({
      success:true,
      readerLevel
    })
  } catch (error) {
    console.error("Error getting reader level:", error);
    res.status(500).json({ error: error.message });
  }
}

module.exports = {
  signup,
  signin,
  updateProfileImage,
  updateCoverImage,
  updateProfileDetails,
  getProfile,
  getUserStats,
  getBasicProfile,
  followUser,
  getStats,
  googleAuth,
 
  getNotifications,
  markNotificationsAsRead,
  createLikeNotification,
  createReviewNotification,
  createReplyNotification,
  notifyFollowers,
  getFollowers,
  getFollowing,
  checkFollowStatus,
  getUserStories,
  getUserSeries,
  getUserProfile,
  updateAccountDetails,

  deleteNotification,

  getCoinBalance,
  redeemCoins,
  getRedemptionRequests,
  processRedemption,
  getRedemptionHistory,
  grantInitialCoinsToAllUsers,
  updateLastActive,
  adminImpersonateUser,
  getReaderWallOfFame,
  updateCoinsSpent,
  
  getUserReaderLevelData,
  getUserReaderLevel,
  updatePhoneNumber,
  checkOnboardingStatus,
};
