const mongoose = require("mongoose");

const replySchema = new mongoose.Schema({
  user: {
    type: mongoose.Schema.Types.ObjectId,
    ref: "User",
    required: true,
  },
  content: {
    type: String,
    required: true,
  },
  mentionedUser: {
    type: mongoose.Schema.Types.ObjectId,
    ref: "User",
  },
  likes: [
    {
      type: mongoose.Schema.Types.ObjectId,
      ref: "User",
    },
  ],
  createdAt: {
    type: Date,
    default: Date.now,
  },
});

const reviewSchema = new mongoose.Schema({
  user: {
    type: mongoose.Schema.Types.ObjectId,
    ref: "User",
    required: true,
  },
  rating: {
    type: Number,
    required: true,
    min: 1,
    max: 5,
  },
  content: {
    type: String,
    required: true,
  },
  likes: [
    {
      type: mongoose.Schema.Types.ObjectId,
      ref: "User",
    },
  ],
  replies: [replySchema],
  createdAt: {
    type: Date,
    default: Date.now,
  },
});

const readHistorySchema = new mongoose.Schema({
  user: {
    type: mongoose.Schema.Types.ObjectId,
    ref: "User",
    required: true,
  },
  lastRead: {
    type: Date,
    default: Date.now,
  },
  readCount: {
    type: Number,
    default: 1,
  },
});

const storySchema = new mongoose.Schema(
  {
    title: {
      type: String,
      required: true,
    },
    content: {
      type: String,
      required: true,
    },
    coverImage: {
      type: String,
    },
    summary: {
      type: String,
    },
    author: {
      type: mongoose.Schema.Types.ObjectId,
      ref: "User",
      required: true,
    },
    storyType: {
      type: String,
      required: true,
    },
    categories: [
      {
        type: String,
      },
    ],
    status: {
      type: String,
      enum: ["published", "draft"],
      default: "draft",
    },
    reads: {
      type: Number,
      default: 0,
    },
    readHistory: [readHistorySchema],
    likes: [
      {
        type: mongoose.Schema.Types.ObjectId,
        ref: "User",
      },
    ],
    reviews: [reviewSchema],
    reviewedUsers: [
      {
        type: mongoose.Schema.Types.ObjectId,
        ref: "User",
      },
    ],
    averageRating: {
      type: Number,
      default: 0,
    },
    uniqueReads: {
      type: Number,
      default: 0,
    },
    totalReads: {
      type: Number,
      default: 0,
    },
    language: { // Add language field
      type: String,
      default: "EN",
      required: true,
    },
  },
  { timestamps: true }
);

storySchema.pre("save", function (next) {
  // Calculate average rating
  if (this.reviews.length > 0) {
    const sum = this.reviews.reduce((acc, review) => acc + review.rating, 0);
    this.averageRating = (sum / this.reviews.length).toFixed(1);
  }

  // Calculate unique reads
  this.uniqueReads = this.readHistory.length;

  // Calculate total reads
  this.totalReads = this.readHistory.reduce(
    (sum, record) => sum + record.readCount,
    0
  );

  next();
});

module.exports = mongoose.model("Story", storySchema);
