const mongoose = require('mongoose');

// Track connection status
let isConnected = false;

// Connection options
const options = {
  useNewUrlParser: true,
  useUnifiedTopology: true,
  serverSelectionTimeoutMS: 5000,
  maxPoolSize: 10 // Maintain up to 10 socket connections
};

// Create a cached connection variable
let cachedConnection = null;

const connectDB = async () => {
  // If we already have a connection, don't connect again
  if (isConnected) {
    console.log('Using existing MongoDB connection');
    return;
  }

  // If we have a cached connection, use it
  if (cachedConnection) {
    console.log('Reusing cached MongoDB connection');
    return cachedConnection;
  }

  try {
    console.log('Connecting to MongoDB...');
    
    // Get MongoDB URI from environment variables
    const dbURI = process.env.MONGODB_URI;
    
    if (!dbURI) {
      throw new Error('MongoDB URI not found in environment variables');
    }
    
    // Create the connection
    const connection = await mongoose.connect(dbURI, options);
    
    // Cache the connection
    cachedConnection = connection;
    
    // Set connection status
    isConnected = true;
    
    console.log(`Connected to MongoDB successfully at: ${new Date().toISOString()}`);
    
    // Add event listeners for connection issues
    mongoose.connection.on('error', (err) => {
      console.error('MongoDB connection error:', err);
      isConnected = false;
    });
    
    mongoose.connection.on('disconnected', () => {
      console.warn('MongoDB disconnected');
      isConnected = false;
    });
    
    // Handle process termination
    process.on('SIGINT', async () => {
      await mongoose.connection.close();
      console.log('MongoDB connection closed due to app termination');
      process.exit(0);
    });
    
    return connection;
  } catch (error) {
    console.error('MongoDB connection error:', error);
    isConnected = false;
    process.exit(1); // Exit with failure
  }
};

// Function to check connection status
const checkConnection = () => {
  return isConnected;
};

module.exports = { connectDB, checkConnection };
