const express = require("express");
const mongoose = require("mongoose");
const cors = require("cors");
const dotenv = require("dotenv");
const routes = require("./routes/api");
const path = require("path");
const compression = require("compression");   
dotenv.config();
const app = express();

// Use compression for all responses with higher compression level
app.use(compression({
  level: 6,
  threshold: 0, // Compress all responses
  filter: (req, res) => {
    if (req.headers['x-no-compression']) {
      return false;
    }
    return compression.filter(req, res);
  }
}));

// Place MongoDB connection at the very start after initializing express
mongoose.connect(process.env.MONGODB_URI, {
  useNewUrlParser: true,
  useUnifiedTopology: true,
  serverSelectionTimeoutMS: 15000, // Reduced from 30000 for faster failure detection
  socketTimeoutMS: 30000, // Reduced from 45000
  family: 4,
  w: 'majority',
  retryWrites: true,
  maxPoolSize: 20, // Increased from 10 for more concurrent connections
  readPreference: 'secondaryPreferred',
  connectTimeoutMS: 15000, // Reduced from 30000
  // Add connection pool monitoring
  monitorCommands: process.env.NODE_ENV === 'development',
  // Add heartbeat to keep connections alive
  heartbeatFrequencyMS: 10000
})
.then(() => {
  console.log("Connected to MongoDB successfully at:", new Date().toISOString());
})
.catch((err) => {
  console.error("MongoDB connection error:", err);
  process.exit(1);
});

// Add connection event listeners for better monitoring
mongoose.connection.on('connected', () => {
  console.log('Mongoose connected to MongoDB');
});

mongoose.connection.on('error', (err) => {
  console.error('Mongoose connection error:', err);
  // Try to reconnect automatically
  if (err.name === 'MongoNetworkTimeoutError' || err.name === 'MongoNetworkError') {
    console.log('Attempting to reconnect to MongoDB...');
    mongoose.disconnect().then(() => {
      mongoose.connect(process.env.MONGODB_URI).catch(err => {
        console.error('Reconnection failed:', err);
      });
    });
  }
});

mongoose.connection.on('disconnected', () => {
  console.log('Mongoose disconnected');
});

// Set request timeouts
app.use((req, res, next) => {
  // Increase timeouts for all requests
  req.setTimeout(30000);
  res.setTimeout(30000);
  next();
});

// Response time monitoring with more detailed logging
app.use((req, res, next) => {
  const start = Date.now();
  res.on('finish', () => {
    const duration = Date.now() - start;
    if (duration > 1000) {
      console.log(`Slow request: ${req.method} ${req.originalUrl} - ${duration}ms`);
      // Log more details for very slow requests
      if (duration > 5000) {
        console.log(`Very slow request details: ${req.method} ${req.originalUrl}`);
        console.log(`Headers: ${JSON.stringify(req.headers)}`);
        console.log(`Query params: ${JSON.stringify(req.query)}`);
      }
    }
  });
  next();
});

// Simplified CORS configuration
const allowedOrigins = [
  'https://novellhub.com',
  'https://www.novellhub.com',
  'http://localhost:5173',
  'http://localhost:3000',
  'http://localhost:5000',
  'https://backend-astroai.com.novellhub.com/',
  'http://localhost:5001',
  'https://novellhub.api.backend.novellhub.com'
];

app.use(cors({
  origin: function(origin, callback) {
    if (!origin || allowedOrigins.indexOf(origin) !== -1) {
      callback(null, true);
    } else {
      callback(null, false);
    }
  },
  credentials: true,
  methods: ['GET', 'POST', 'PUT', 'DELETE', 'OPTIONS', 'PATCH'],
  allowedHeaders: ['Content-Type', 'Authorization', 'X-Requested-With', 'Accept', 'Origin', 'Cache-Control', 'Pragma', 'Expires'],
  exposedHeaders: ['Content-Type', 'Authorization']
}));

// Handle preflight requests explicitly
app.options('*', cors());

// Add these headers to every response
app.use((req, res, next) => {
  res.header('Access-Control-Allow-Credentials', 'true');
  res.header('Access-Control-Allow-Methods', 'GET,PUT,POST,DELETE,OPTIONS,PATCH');
  res.header('Access-Control-Allow-Headers', 'Origin, X-Requested-With, Content-Type, Accept, Authorization, Cache-Control, Pragma, Expires');
  // Enable strong ETags and caching headers
  res.setHeader('ETag', true);
  
  // Add caching headers for GET requests
  if (req.method === 'GET') {
    // Cache static resources longer
    if (req.url.match(/\.(jpg|jpeg|png|gif|css|js)$/)) {
      res.setHeader('Cache-Control', 'public, max-age=86400'); // 1 day
    } else {
      res.setHeader('Cache-Control', 'public, max-age=300'); // 5 minutes
    }
  }
  
  next();
});

// Add static file serving with caching
app.use(express.static('public', {
  maxAge: '1d', // Cache static assets for 1 day
  etag: true,
  lastModified: true
}));

// Add a specific route for favicon
app.get('/favicon.ico', (req, res) => {
  res.sendFile(path.join(__dirname, 'public', 'favicon.ico'));
});

app.get('/.well-known/assetlinks.json', (req, res) => {
  res.setHeader('Content-Type', 'application/json');
  res.sendFile(path.join(__dirname, 'public', '.well-known', 'assetlinks.json'));
});

app.use((req, res, next) => {
  res.header("Access-Control-Allow-Origin", req.headers.origin);
  res.header("Access-Control-Allow-Credentials", true);
  res.header("Cross-Origin-Opener-Policy", "same-origin-allow-popups");
  res.header("Cross-Origin-Embedder-Policy", "require-corp");
  
  // Handle preflight requests
  if (req.method === 'OPTIONS') {
    res.status(204).end();
    return;
  }
  next();
});

// Configure JSON parsing with optimized settings
app.use(express.json({ 
  limit: "10mb", // Reduced from 50mb to prevent large payload attacks
  strict: false
}));
app.use(express.urlencoded({ 
  limit: "10mb", // Reduced from 50mb
  extended: true,
  parameterLimit: 5000 // Reduced from 50000
}));

// Add error handling middleware
app.use((err, req, res, next) => {
  console.error(err.stack);
  res.status(500).json({
    message: 'Something went wrong!',
    error: process.env.NODE_ENV === 'development' ? err.message : {}
  });
});

// Simple test endpoint
app.get("/help", (req,res) =>{
  res.send(process.env.test);
});

// Optimized health check endpoint
app.get("/health", (req, res) => {
  const mongoStatus = mongoose.connection.readyState;
  const status = {
    server: "running",
    mongodb: mongoStatus === 1 ? "connected" : "disconnected",
    timestamp: new Date(),
    uptime: process.uptime()
  };
  
  res.json(status);
});

// Simplified API health endpoint
app.get('/api/health', (req, res) => {
  res.status(200).json({ status: 'OK' });
});

// Add uploads directory with caching
app.use("/uploads", express.static("uploads", {
  maxAge: '1d',
  etag: true,
  lastModified: true
}));

// Use routes
app.use("/", routes);

// Start server
const PORT = process.env.PORT || 5000;
const server = app.listen(PORT, () => {
  console.log(`Server running on port ${PORT}`);
});

// Add server timeout handling
server.timeout = 60000; // 60 seconds

// Add graceful shutdown
process.on('SIGTERM', () => {
  console.log('SIGTERM signal received: closing HTTP server');
  server.close(() => {
    console.log('HTTP server closed');
    mongoose.connection.close(false, () => {
      console.log('MongoDB connection closed');
      process.exit(0);
    });
  });
});