const jwt = require('jsonwebtoken');
const User = require('../Model/User_Model');
const Admin = require('../Model/Admin/Admin_Model');

const auth = async (req, res, next) => {
  try {
    const token = req.headers.authorization?.split(' ')[1];
    if (!token) {
      return res.status(401).json({ error: 'No token provided' });
    }

    const decoded = jwt.verify(token, process.env.JWT_SECRET);
    const user = await User.findById(decoded.userId);
    
    if (!user) {
      return res.status(401).json({ error: 'User not found' });
    }

    req.userId = decoded.userId;
    req.user = user;
    next();
  } catch (error) {
    res.status(401).json({ error: 'Authentication failed' });
  }
};

// Restore the original adminAuth middleware
const adminAuth = async (req, res, next) => {
  try {
    const token = req.header('Authorization')?.replace('Bearer ', '');
    
    if (!token) {
      return res.status(401).json({ error: 'Authentication required' });
    }
    
    const decoded = jwt.verify(token, process.env.JWT_SECRET);
    
    // Check if this is an admin token
    const admin = await Admin.findById(decoded.adminId);
    
    if (!admin) {
      return res.status(403).json({ error: 'Unauthorized access' });
    }
    
    req.adminId = decoded.adminId;
    req.admin = admin;
    
    next();
  } catch (error) {
    console.error("Admin auth error:", error);
    res.status(401).json({ error: 'Authentication failed' });
  }
};

// Add a new middleware for admin impersonation
const adminImpersonationAuth = async (req, res, next) => {
  try {
    const token = req.header('Authorization')?.replace('Bearer ', '');
    
    if (!token) {
      return res.status(401).json({ error: 'Authentication required' });
    }
    
    const decoded = jwt.verify(token, process.env.JWT_SECRET);
    console.log("Decoded token for impersonation:", decoded);
    
    // Check if this is an admin token
    if (decoded.adminId) {
      // This is an admin token
      const admin = await Admin.findById(decoded.adminId);
      
      if (!admin) {
        return res.status(403).json({ error: 'Unauthorized access' });
      }
      
      req.adminId = decoded.adminId;
      req.admin = admin;
      req.isAdmin = true;
    } else {
      // This is a user token, check if user has admin role
      const user = await User.findById(decoded.userId);
      
      if (!user) {
        return res.status(401).json({ error: 'User not found' });
      }
      
      if (user.role !== 'admin') {
        return res.status(403).json({ error: 'Unauthorized access' });
      }
      
      req.userId = decoded.userId;
      req.user = user;
      req.isAdmin = true;
    }
    
    next();
  } catch (error) {
    console.error("Admin impersonation auth error:", error);
    res.status(401).json({ error: 'Authentication failed' });
  }
};

module.exports = { auth, adminAuth, adminImpersonationAuth };
