const User = require("../Model/User_Model");

const isDuplicateNotification = async (userId, notification) => {
  const user = await User.findById(userId);
  const fiveMinutesAgo = new Date(Date.now() - 5 * 60 * 1000); // Increased window to 5 minutes

  return user.notifications.some(
    (notif) =>
      notif.type === notification.type &&
      // For contact_reply and level_up notifications, fromUser is null, so skip this check
      (notification.type === "contact_reply" || notification.type === "level_up" || 
       notif.fromUser?.toString() === notification.fromUser?.toString()) &&
      (notification.type === "follow" || notification.type === "contact_reply" || notification.type === "level_up" ||
        notif.contentId?.toString() === notification.contentId?.toString()) &&
      new Date(notif.createdAt) > fiveMinutesAgo
  );
};


const createNotification = async (userId, notification) => {
  try {
    // Skip if sender is same as recipient (except for system notifications like contact_reply and level_up)
    if (notification.type !== "contact_reply" && notification.type !== "level_up" && 
        userId.toString() === notification.fromUser?.toString()) {
      return;
    }

    // Skip if content author is the reviewer
    if (notification.contentAuthorId?.toString() === notification.fromUser?.toString()) {
      return;
    }

    // Skip if review author is replying to their own review
    if (notification.reviewAuthorId?.toString() === notification.fromUser?.toString()) {
      return;
    }

    // For follow notifications, ensure contentId and contentModel are set
    if (notification.type === "follow") {
      // If contentId is null, use the fromUser as the contentId
      if (!notification.contentId) {
        notification.contentId = notification.fromUser;
      }
      
      // If contentModel is null, set it to "User"
      if (!notification.contentModel) {
        notification.contentModel = "User";
      }
    }

    // For level_up notifications, ensure contentId and contentModel are set
    if (notification.type === "level_up") {
      // If contentId is null, use the userId as the contentId
      if (!notification.contentId) {
        notification.contentId = userId;
      }
      
      // If contentModel is null, set it to "User"
      if (!notification.contentModel) {
        notification.contentModel = "User";
      }
    }

    // For tip notifications, ensure contentId and contentModel are set
    if (notification.type === "tip") {
      // Make sure we have the content information
      if (!notification.contentId || !notification.contentModel) {
        console.log("Skipping tip notification creation: contentId or contentModel is missing");
        return;
      }
    }

    // For unlock notifications, ensure contentId and contentModel are set
    if (notification.type === "unlock") {
      // Make sure we have the content information
      if (!notification.contentId || !notification.contentModel) {
        console.log("Skipping unlock notification creation: contentId or contentModel is missing");
        return;
      }
    }

    // For all notifications, ensure contentId and contentModel are not null
    // If they would be null, don't create the notification
    if (!notification.contentId || !notification.contentModel) {
      console.log("Skipping notification creation: contentId or contentModel is missing", {
        type: notification.type,
        contentId: notification.contentId,
        contentModel: notification.contentModel
      });
      return;
    }

    const isDuplicate = await isDuplicateNotification(userId, notification);
    if (isDuplicate) {
      return;
    }

    console.log(`Adding notification to user ${userId}:`, {
      type: notification.type,
      message: notification.message,
      contentId: notification.contentId
    });

    const result = await User.findByIdAndUpdate(userId, {
      $push: {
        notifications: {
          type: notification.type,
          message: notification.message,
          fromUser: notification.fromUser,
          contentId: notification.contentId,
          contentModel: notification.contentModel,
          createdAt: new Date(),
          isRead: false,
          paymentStatus: notification.paymentStatus,
          paymentAmount: notification.paymentAmount,
          additionalData: notification.additionalData
        }
      }
    });

    console.log("Update result:", result ? "Success" : "Failed");
  } catch (error) {
    console.error('Notification creation error:', error);
  }
};


// Create a level-up notification
const createLevelUpNotification = async (userId, previousLevel, newLevel, bonusCoins) => {
  try {
    const user = await User.findById(userId);
    if (!user) {
      console.log(`User ${userId} not found for level-up notification`);
      return false;
    }

    const message = `Congratulations! You've reached ${newLevel} Reader level and earned ${bonusCoins} bonus coins!`;
    
    const notification = {
      type: "level_up",
      message: message,
      fromUser: null, // System notification
      contentId: userId,
      contentModel: "User",
      createdAt: new Date(),
      isRead: false,
      additionalData: {
        previousLevel,
        newLevel,
        bonusCoins,
        timestamp: new Date().toISOString()
      }
    };

    console.log(`Creating level-up notification for user ${userId}:`, notification);

    const result = await User.findByIdAndUpdate(
      userId,
      { $push: { notifications: { $each: [notification], $position: 0 } } },
      { new: true }
    );

    console.log(`Level-up notification created: ${!!result}`);
    return true;
  } catch (error) {
    console.error('Error creating level-up notification:', error);
    return false;
  }
};

const notifyFollowers = async (authorId, contentId, contentType, title) => {
  try {
    const author = await User.findById(authorId).populate("followers");
    if (!author || !author.followers.length) return;

    const notifications = author.followers.map((follower) => ({
      userId: follower._id,
      notification: {
        type: "new_content",
        message: `${
          author.username
        } published a new ${contentType.toLowerCase()} "${title}"`,
        fromUser: authorId,
        contentId,
        contentModel: contentType,
        isRead: false,
      },
    }));

    await Promise.all(
      notifications.map(({ userId, notification }) =>
        createNotification(userId, notification)
      )
    );
  } catch (error) {
    console.error("Notify followers error:", error);
  }
};

const removeContentNotifications = async (contentId) => {
  try {
    await User.updateMany(
      { "notifications.contentId": contentId },
      { $pull: { notifications: { contentId: contentId } } }
    );
  } catch (error) {
    console.error("Error removing content notifications:", error);
  }
};

const removeNotification = async (userId, notificationId) => {
  try {
    await User.findByIdAndUpdate(userId, {
      $pull: { notifications: { _id: notificationId } },
    });
  } catch (error) {
    console.error("Error removing notification:", error);
  }
};

// Create a payment notification
const createPaymentNotification = async (userId, status, amount) => {
  try {
    const message = status === 'approved' 
      ? `Your payment request for ₹${amount} has been approved and will be processed shortly.`
      : `Your payment request for ₹${amount} has been rejected. Please contact support for more information.`;
    
    await User.findByIdAndUpdate(userId, {
      $push: {
        notifications: {
          type: "payment",
          message: message,
          fromUser: null, // Admin notification doesn't need a fromUser
          contentId: null,
          contentModel: null,
          createdAt: new Date(),
          isRead: false,
          paymentStatus: status,
          paymentAmount: amount
        }
      }
    });
  } catch (error) {
    console.error('Payment notification creation error:', error);
  }
};

// Create a notification for contact form replies
const createContactReplyNotification = async (userId, contactId, subject, reply) => {
  try {
    if (!userId) {
      console.log("No userId provided for contact reply notification");
      return false;
    }
    
    console.log(`Creating contact reply notification for user ${userId}`);
    
    // Check if user exists
    const user = await User.findById(userId);
    if (!user) {
      console.log(`User ${userId} not found`);
      return false;
    }
    
    // Create notification directly without using createNotification
    // This bypasses some of the checks that might be causing issues
    const notificationData = {
      type: "contact_reply",
      message: `We've responded to your message: "${subject}"`,
      fromUser: null, // Admin/system notification
      contentId: contactId,
      contentModel: "Contact",
      createdAt: new Date(),
      isRead: false,
      additionalData: {
        reply: reply,
        originalSubject: subject
      }
    };
    
    console.log("Adding notification directly:", notificationData);
    
    const result = await User.findByIdAndUpdate(
      userId,
      { $push: { notifications: notificationData } },
      { new: true }
    );
    
    console.log("Notification added successfully:", !!result);
    return true;
  } catch (error) {
    console.error("Error creating contact reply notification:", error);
    return false;
  }
};

module.exports = {
  createNotification,
  notifyFollowers,
  isDuplicateNotification,
  removeContentNotifications,
  removeNotification,
  createPaymentNotification,
  createContactReplyNotification,
  createLevelUpNotification
};
