const User = require("../Model/User_Model");
const { createNotification } = require("./notifications");

const updateReaderLevel = async (userId, coinsToAdd = 0) => {
    try {
      // Get the user
      const user = await User.findById(userId);
      if (!user) return null;
      
      // Store the previous level for comparison
      const previousLevel = user.readerLevel || 'Bronze';
      
      // Update coins spent
      const newCoinsSpent = (user.coinsSpent || 0) + coinsToAdd;
      
      // Determine the new level based on coins spent
      let newLevel = 'Bronze';
      let updatedBadges = { ...user.badges || { bronze: true, silver: false, gold: false, platinum: false } };
      
      if (newCoinsSpent >= 1200) {
        newLevel = 'Platinum';
        updatedBadges = { bronze: true, silver: true, gold: true, platinum: true };
      } else if (newCoinsSpent >= 500) {
        newLevel = 'Gold';
        updatedBadges = { bronze: true, silver: true, gold: true, platinum: false };
      } else if (newCoinsSpent >= 200) {
        newLevel = 'Silver';
        updatedBadges = { bronze: true, silver: true, gold: false, platinum: false };
      }
      
      // Check if user leveled up to award bonus coins
      let bonusCoins = 0;
      let levelUpMessage = null;
      
      if (newLevel !== previousLevel) {
        // User leveled up - determine bonus coins
        if (previousLevel === 'Bronze' && newLevel === 'Silver') {
          bonusCoins = 50;
          levelUpMessage = `Congratulations! You've reached Silver Reader level and earned ${bonusCoins} bonus coins!`;
        } else if (previousLevel === 'Silver' && newLevel === 'Gold') {
          bonusCoins = 100;
          levelUpMessage = `Congratulations! You've reached Gold Reader level and earned ${bonusCoins} bonus coins!`;
        } else if (previousLevel === 'Gold' && newLevel === 'Platinum') {
          bonusCoins = 350;
          levelUpMessage = `Congratulations! You've reached Platinum Reader level and earned ${bonusCoins} bonus coins!`;
        }
      }
      
      // Update user with new level, badges, and bonus coins if applicable
      const updateData = { 
        readerLevel: newLevel, 
        badges: updatedBadges,
        coinsSpent: newCoinsSpent
      };
      
      // Add bonus coins if user leveled up
      if (bonusCoins > 0) {
        updateData.$inc = { coins: bonusCoins };
      }
      
      const updatedUser = await User.findByIdAndUpdate(
        userId,
        updateData,
        { new: true }
      );
      
// Inside the updateReaderLevel function where level-up is detected:

if (newLevel !== previousLevel) {
    try {
      // Create a notification for the user about level up
      await User.findByIdAndUpdate(userId, {
        $push: {
          notifications: {
            type: "level_up",
            message: levelUpMessage || `Congratulations! You've reached ${newLevel} Reader level!`,
            fromUser: null,
            contentId: userId,
            contentModel: "User",
            createdAt: new Date(),
            isRead: false,
            additionalData: {
              previousLevel,
              newLevel,
              bonusCoins
            }
          }
        }
      });
      
      console.log(`Level-up notification created for user ${userId}: ${levelUpMessage}`);
    } catch (notifError) {
      console.error("Failed to create level-up notification:", notifError);
    }
  }
  
      
      return {
        level: updatedUser.readerLevel,
        coinsSpent: updatedUser.coinsSpent,
        badges: updatedUser.badges,
        levelUp: newLevel !== previousLevel,
        bonusCoins: bonusCoins
      };
    } catch (error) {
      console.error("Error updating reader level:", error);
      return null;
    }
  };


  
  const getReaderLevelInfo = async (userId) => {
    try {
      const user = await User.findById(userId);
      if (!user) return null;
      
      const coinsSpent = user.coinsSpent || 0;
      let currentLevel = {
        level: 'Bronze',
        badge: '🥉',
        perks: [
          'Earn 50 Coins for completing Bronze level',
        ]
      };
      
      let nextLevel = {
        level: 'Silver',
        coinsNeeded: 200 - coinsSpent,
        reward: 50,
        perks: [
         'Earn 100 Coins for completing Silver level'
        ]
      };
      
      let progress = 0;
      
      if (coinsSpent >= 1200) {
        currentLevel = {
          level: 'Platinum',
          badge: '💎',
          perks: [
            'Earned 350 Coins for reaching max level',
            'This is the maximum reader level'
          ]
        };
        nextLevel = null;
        progress = 100;
      } else if (coinsSpent >= 500) {
        currentLevel = {
          level: 'Gold',
          badge: '🥇',
          perks: [
            'Earn 350 Coins for completing Gold level'
          ]
        };
        nextLevel = {
          level: 'Platinum',
          coinsNeeded: 1200 - coinsSpent,
          reward: 350,
          perks: [
            'Earn 350 Coins for reaching max level',
            'This is the maximum reader level'
          ]
        };
        progress = Math.min(100, Math.floor((coinsSpent - 500) / 7));
      } else if (coinsSpent >= 200) {
        currentLevel = {
          level: 'Silver',
          badge: '🥈',
          perks: [
            'Earn 100 Coins for completing Silver level'
          ]
        };
        nextLevel = {
          level: 'Gold',
          coinsNeeded: 500 - coinsSpent,
          reward: 200,
          perks: [
            'Earn 200 Coins for completing Gold level'
          ]
        };
        progress = Math.min(100, Math.floor((coinsSpent - 200) / 3));
      } else {
        progress = Math.min(100, Math.floor(coinsSpent / 2));
      }
      
      return {
        current: currentLevel,
        nextLevel,
        progress,
        coinsSpent
      };
    } catch (error) {
      console.error("Error getting reader level info:", error);
      return null;
    }
  };

  


  const getReaderLevel = async (req, res) => {
    try {
      const userId = req.userId;
      
      // Find the user
      const user = await User.findById(userId);
      if (!user) {
        return res.status(404).json({ error: "User not found" });
      }
      
      // Get detailed reader level info
      const readerLevelInfo = await getReaderLevelInfo(userId);
      
      // Return reader level info
      res.status(200).json({
        success: true,
        readerLevel: readerLevelInfo
      });
    } catch (error) {
      console.error("Error getting reader level:", error);
      res.status(500).json({ error: error.message });
    }
  };


module.exports = {
  updateReaderLevel,
  getReaderLevelInfo,
  getReaderLevel
};
